#!/usr/bin/python

########################################################################
#
#  A Python script for controlling the Adafruit 1.8" TFT LCD module
#  from a Raspbery Pi.
#
#   Author :   Bruce E. Hall, W8BH    <bhall66@gmail.com>
#   Date   :   25 Feb 2014
#
#   This module uses the ST7735 controller and SPI data interface
#   PART 4 --- TEXT
#
#   For more information, see w8bh.net
#
########################################################################

import cooper14 as font
import font5x7 as font0
import RPi.GPIO as GPIO
import time
import os                      #for popen
import spidev                  #hardware PI
from random import randint
from math   import sqrt
import json
import MFRC522

# Create an object of the class MFRC522
MIFAREReader = MFRC522.MFRC522()

data = {}

#TFT to RPi connections
# PIN TFT      RPi
# 1 GND		GND
# 2 VCC	   	3,3V
# 3 SCK    	SCLK (GPIO 11)
# 4 MOSI    	MOSI (GPIO 10)
# 5 RES    	RST  (GPIO 25)
# 6 D/C		GPIO 24
# 7 CS    	CE1  (GPIO 08)
# 8 LEDA    	3,3V (GPIO 17)


#DC      = 24 #  GPIO.BCM
DC      = 18 # GPIO.BOARD
LEDA    = 11
XSIZE   = 128
YSIZE   = 160
XMAX    = XSIZE-1
YMAX    = YSIZE-1
X0      = XSIZE/2
Y0      = YSIZE/2

#Color constants
BLACK   = 0x0000
BLUE    = 0x001F
RED     = 0xF800
GREEN   = 0x0400
LIME    = 0x07E0
CYAN    = 0x07FF
MAGENTA = 0xF81F
YELLOW  = 0xFFE0
WHITE   = 0xFFFF
PURPLE  = 0x8010
NAVY    = 0x0010
TEAL    = 0x0410
OLIVE   = 0x8400
MAROON  = 0x8000
SILVER  = 0xC618
GRAY    = 0x8410

bColor  = BLACK
fColor  = YELLOW

COLORSET  = [BLACK,BLUE,RED,GREEN,LIME,CYAN,MAGENTA,YELLOW,
             WHITE,PURPLE,NAVY,TEAL,OLIVE,MAROON,SILVER,GRAY]

#TFT display constants
SWRESET = 0x01
SLPIN   = 0x10
SLPOUT  = 0x11
PTLON   = 0x12
NORON   = 0x13
INVOFF  = 0x20
INVON   = 0x21
DISPOFF = 0x28
DISPON  = 0x29
CASET   = 0x2A
RASET   = 0x2B
RAMWR   = 0x2C
RAMRD   = 0x2E
PTLAR   = 0x30
MADCTL  = 0x36
COLMOD  = 0x3A


########################################################################
#
#   Low-level routines
#
#

def SetPin(pinNumber,value):
    #sets the GPIO pin to desired value (1=on,0=off)
    GPIO.output(pinNumber,value)

def InitGPIO():
#    GPIO.setmode(GPIO.BCM)
    GPIO.setmode(GPIO.BOARD)
    GPIO.setwarnings(False)
    GPIO.setup(DC,GPIO.OUT)
# Pin 11 (GPIO 17) fuer LED-Hintergrundbeleuchtung als Ausgang festlegen
    GPIO.setup(LEDA, GPIO.OUT)

def InitSPI():
  "returns an opened spi connection to device(0,1) in mode 0"
  spiObject = spidev.SpiDev()
  spiObject.open(0,1)
  spiObject.max_speed_hz =16000000
  spiObject.mode = 0
  return spiObject


########################################################################
#
#   ST7735 TFT controller routines:
#
#

def WriteByte(value):
  "sends an 8-bit value to the display as data"
  SetPin(DC,1)
  spi.writebytes([value])

def WriteWord (value):
  "sends a 16-bit value to the display as data"
  SetPin(DC,1)
  spi.writebytes([value>>8, value&0xFF])

def Command(cmd, *bytes):
  "Sends a command followed by any data it requires"
  SetPin(DC,0)                    #command follows
  spi.writebytes([cmd])           #send the command byte
  if len(bytes)>0:                #is there data to follow command?
    SetPin(DC,1)                  #data follows
    spi.writebytes(list(bytes))   #send the data bytes
     
def InitDisplay():
  "Resets & prepares display for active use."
  Command (SWRESET)               #reset TFT controller
  time.sleep(0.2)         	  #wait 200mS for controller init
  Command (SLPOUT)                #wake from sleep
  Command (COLMOD, 0x05)          #set color mode to 16 bit
  Command (DISPON)                #turn on display

def SetOrientation(degrees):
  "Set the display orientation to 0,90,180,or 270 degrees"
  if   degrees==90:  arg=0x60
  elif degrees==180: arg=0xC0
  elif degrees==270: arg=0xA0
  else:              arg=0x00
  Command (MADCTL,arg)

def SetAddrWindow(x0,y0,x1,y1):
  "sets a rectangular display window into which pixel data is placed"
  Command (CASET,0,x0,0,x1)       #set column range (x0,x1)
  Command (RASET,0,y0,0,y1)       #set row range (y0,y1)

def WriteBulk (value,reps,count=1):
  "sends a 16-bit pixel word many, many times using hardware SPI"
  "number of writes = reps * count. Value of reps must be <= 2048"
  SetPin(DC,0)                    #command follows
  spi.writebytes([RAMWR])         #issue RAM write command
  SetPin(DC,1)                    #data follows
  valHi = value >> 8              #separate color into two bytes
  valLo = value & 0xFF
  byteArray = [valHi,valLo]*reps  #create buffer of multiple pixels
  for a in range(count):
    spi.writebytes(byteArray)   #send this buffer multiple times

def WritePixels (byteArray):
  "sends pixel data to the TFT"
  SetPin(DC,0)                    #command follows
  spi.writebytes([RAMWR])         #issue RAM write command
  SetPin(DC,1)                    #data follows
  spi.writebytes(byteArray)       #send data to the TFT


########################################################################
#
#   Graphics routines:
#
#


def FillRect(x0,y0,x1,y1,color):
  "fills rectangle with given color"
  width = x1-x0+1
  height = y1-y0+1
  SetAddrWindow(x0,y0,x1,y1)
  WriteBulk(color,width,height)

def FillScreen(color):
  "Fills entire screen with given color"
  FillRect(0,0,XMAX,YMAX,color)

def ClearScreen():
  "Fills entire screen with black"
  FillScreen(BLACK)


########################################################################
#
#   Text routines:
#
#

def GetCharData (ch):
  "Returns array of raster data for a given ASCII character"
  pIndex = ord(ch)-ord(' ')
  lastDescriptor = len(font.descriptor)-1
  charIndex = font.descriptor[pIndex][1]
  if (pIndex >= lastDescriptor):
    return font.rasterData[charIndex:]
  else:
    nextIndex = font.descriptor[pIndex+1][1]
    return font.rasterData[charIndex:nextIndex]

def GetCharWidth(ch):
  "returns the width of a character, in pixels"
  pIndex = ord(ch)-ord(' ')
  return font.descriptor[pIndex][0]

def GetCharHeight(ch):
  "returns the height of a character, in pixels"
  return font.fontInfo[0]

def GetStringWidth(st):
  "returns the width of the text, in pixels"
  width = 0
  for ch in st:
    width += GetCharWidth(ch) + 1
  return width

def PutCh (ch,xPos,yPos,color=fColor):
  "write ch to X,Y coordinates using ASCII 5x7 font"
  charData = font0.data[ord(ch)-32]
  SetAddrWindow(xPos,yPos,xPos+4,yPos+6)
  SetPin(DC,0)
  spi.writebytes([RAMWR])
  SetPin(DC,1)
  buf = []
  mask = 0x01
  for row in range(7):
    for col in range(5):
      bit = charData[col] & mask
      if (bit==0):
        pixel = bColor
      else:
        pixel = color
      buf.append(pixel>>8)
      buf.append(pixel&0xFF)
    mask <<= 1
  spi.writebytes(buf)

def PutCharV2 (ch,xPos,yPos,color=fColor):
  "Writes Ch to X,Y coordinates in current foreground color"
  charData = GetCharData(ch)
  xLen = GetCharWidth(ch)
  numRows = GetCharHeight(ch)
  bytesPerRow = 1+((xLen-1)/8)
  numBytes = numRows*bytesPerRow
  SetAddrWindow(xPos,yPos,xPos+xLen-1,yPos+numRows-1)
  SetPin(DC,0)
  spi.writebytes([RAMWR])
  SetPin(DC,1)
  i= 0
  while (i< numBytes):
    mask = 0x01
    rowBits = 0
    for b in range(bytesPerRow):
      rowBits <<= 8
      mask    <<= 8
      rowBits +=  charData[i]
      i +=  1
    mask >>= 1
    #at this point, all bits for current row should
    #be in rowBits, regardless of number of bytes
    #it takes to represent the row

    rowBuf = []
    for a in range(xLen):
      bit = rowBits & mask
      mask >>= 1
      if (bit==0):
        pixel = bColor
      else:
        pixel = color
      rowBuf.append(pixel>>8)
      rowBuf.append(pixel&0xFF)
    spi.writebytes(rowBuf)

def PutChar (ch,xPos,yPos,color=fColor):
  "Writes Ch to X,Y coordinates in current foreground color"
  charData = GetCharData(ch)
  xLen = GetCharWidth(ch)                   #char width, in pixels
  numRows = GetCharHeight(ch)
  bytesPerRow = 1+((xLen-1)/8)              #char width, in bytes
  numBytes = numRows*bytesPerRow
  SetAddrWindow(xPos,yPos,xPos+xLen-1,yPos+numRows-1)
  SetPin(DC,0)
  spi.writebytes([RAMWR])                   #pixel data to follow
  SetPin(DC,1)
  index = 0
  buf = []
  for a in range(numRows):                  #row major
    bitNum = 0                   
    for b in range(bytesPerRow):          #do whole row
      mask = 0x80                        #msb first
      for c in range(8):                 #do all bits in this byte
        if (bitNum<xLen):              #still within char width?
          bit = charData[index] & mask
          if (bit==0):                #check the bit
            pixel = bColor         #0: background color
          else:
            pixel = color           #1: foreground color
          buf.append(pixel>>8)        #add pixel to buffer
          buf.append(pixel&0xFF)
          mask >>= 1                  #goto next bit in byte
        bitNum += 1                    #goto next bit in row
      index += 1                        #goto next byte of data
  spi.writebytes(buf)                      #send char data to TFT
     
def PutString(xPos,yPos,st,color=fColor):
  "Draws string on display at xPos,yPos."
  "Does NOT check to see if it fits!"
  for ch in st:
    width = GetCharWidth(ch)+1
    PutChar(ch,xPos,yPos,color)
    xPos += width


########################################################################
#
#   Testing routines:
#
#

def PrintElapsedTime(function,startTime):
  "Formats an output string showing elapsed time since function start"
  elapsedTime=time.time()-startTime
  print "%15s: %8.3f seconds" % (function,elapsedTime)
  time.sleep(1)

def ColorBars():
  "Fill Screen with 8 color bars"
  for a in range(8):
    FillRect(0,a*20,XMAX,a*20+19,COLORSET[a+1])

def ScreenTest():
  "Measures time required to fill display twice"
  startTime=time.time()
  FillScreen(LIME)
  FillScreen(MAGENTA)
  ColorBars()
  PrintElapsedTime('ScreenTest',startTime)


def PortaitChars():
  "Writes 420 characters (5x7) to screen in Portrait Mode"
  #font is 5x7 with 1 pixel spacing
  #so character width = 6 pixels, height = 8 pixels
  #display width = 128 pixels, so 21 char per row (21x6 = 126)
  #display ht = 160 pixels, so 20 rows (20x8 = 160)
  #total number of characters = 21 x 20 = 420

  CHARS_PER_ROW = 21
  FillRect(0,0,XMAX,YMAX,bColor)  #clear screen
  for i in range(420):
    x= i % CHARS_PER_ROW
    y= i / CHARS_PER_ROW
    ascii = (i % 96)+32
    PutCh(chr(ascii),x*6,y*8)
  time.sleep(1)

def LandscapeChars():
  "Writes 416 characters (5x7) to screen, landscape mode"
  #font is 5x7 with 1 pixel spacing
  #so character width = 6 pixels, height = 8 pixels
  #display width = 160 pixels, so 26 char per row (26x6 = 156)
  #display ht = 128 pixels, so 16 rows (16x8 = 128)
  #total number of characters = 26 x 16 = 416

  CHARS_PER_ROW = 26
  FillRect(0,0,YMAX,XMAX,bColor)  #clear screen
  for i in range(416):
    x= i % CHARS_PER_ROW
    y= i / CHARS_PER_ROW
    ascii = (i % 96)+32
    PutCh(chr(ascii),x*6,y*8,CYAN)
  time.sleep(1)
     
def LargeFontTest():
  "Writes 90 characters (11x17) to the screen"
  title = 'Large Font'
  startTime = time.time()
  for i in range(90):
    x= i % 10
    y= i / 10
    ascii = (i % 96)+32
    PutChar(chr(ascii),x*12,y*18,LIME)
  PrintElapsedTime(title,startTime)

def RandColor():
  "Returns a random color from BGR565 Colorspace"
  index = randint(0,len(COLORSET)-1)
  return COLORSET[index]

def SmallFontTest():
  "Writes 2000 random 5x7 characters to the screen"
  title = 'Small Font'
  startTime = time.time()
  for i in range(2000):
    x= randint(0,20)
    y= randint(0,19)
    color = RandColor()
    ascii = (i % 96)+32
    PutCh(chr(ascii),x*6,y*8,color)
  PrintElapsedTime(title,startTime)

def OrientationTest():
  "Write 5x7 characters at 0,90,180,270 deg orientations"
  title = 'Orientation'
  startTime = time.time()
  PortaitChars()                     
  SetOrientation(90)              #display-top on right
  LandscapeChars()
  SetOrientation(180)             #upside-down
  PortaitChars()
  SetOrientation(270)             #display-top on left   LandscapeChars()
  SetOrientation(0)               #return to 0 deg.
  PrintElapsedTime(title,startTime)

def GetTempCPU():
  "Returns CPU temp in degrees F"
  tPath = '/sys/class/thermal/thermal_zone0/temp'
  tFile = open(tPath)
  temp = tFile.read()
  tFile.close()
  return (float(temp)*0.001)

def Run(cmd):
  "Runs a system (bash) command"
  return os.popen(cmd).read()

def GetIPAddr():
  "Returns IP address as a string"
  cmd = "ifconfig | awk '/192/ {print $2}'"
  res = Run(cmd).replace('\n','') #remove end of line char
  return res.replace('addr:','')  #remove 'addr:' prefix

def ShowInfo():
  "Show IP address, CPU temp, and Time"
  title = 'Info'
#  startTime = time.time()
  SetOrientation(0)
  ClearScreen()
  SetOrientation(90)             #upside-down
  PutString(0,0,'IP addr',WHITE)
  PutString(0,20,GetIPAddr())
  PutString(0,45,'CPU temp',WHITE)
  temp = GetTempCPU()
  PutString(0,65,'{:5.1f} deg C'.format(temp))
  tStr = time.strftime("%H:%M:%S  ")
  PutString(0,90,'Time',WHITE)
  PutString(0,110,tStr)
#  PrintElapsedTime(title,startTime)

#def RunTextTests():
#  startTime = time.time()         #keep track of test duration
#  ScreenTest()
#  LargeFontTest()
#  SmallFontTest()
#  OrientationTest()
#  ClearScreen()
#  InfoTest()
#  PrintElapsedTime('Full Suite',startTime)

def GetJson(jfile):
  global data
  with open(jfile, "r") as json_file:
    data = json.load(json_file)
  json_file.close()

def ShowBMP388():
  "Show Pressure, Temperatur outside, and Altitude"
  title = 'BMP388'
  GetJson('weather-bmp388.json')
  SetOrientation(0)
  ClearScreen()
  SetOrientation(90)             #upside-down
  for p in data['BMP388']:
    PutString(0,0,'Luftdruck',WHITE)
    PutString(0,20,'{:4.1f} hPa'.format(p['Pressure']))
    PutString(0,45,'Temperatur',WHITE)
    PutString(0,65,'{:5.1f} deg C'.format(p['Temperature']))
    PutString(0,90,'Hoehe ueber N.N.',WHITE)
    PutString(0,110,'{:5.1f} m'.format(p['Altitude']))

def ShowMCP9808():
  "Show Temperatur inside"
  title = 'MCP9808'
  GetJson('weather-mcp9808.json')
  SetOrientation(0)
  ClearScreen()
  SetOrientation(90)             #upside-down
  for p in data['MCP9808']:
    PutString(0,0,'Temperatur',WHITE)
    PutString(0,20,'(indoor)',WHITE)
    PutString(0,40,'{:5.1f} deg C'.format(p['Temperature']))


def ShowVEML6070():
  "Show UV-Raw, UV-A, UV-A Index"
  title = 'VEML6070'
  GetJson('weather-veml6070.json')
  SetOrientation(0)
  ClearScreen()
  SetOrientation(90)             #upside-down
  for p in data['VEML6070']:
    PutString(0,0,'UV-Raw',WHITE)
    PutString(0,20,'{:4.1f} W/m2'.format(p['UV-Raw']))
    PutString(0,45,'UV-A',WHITE)
    PutString(0,65,'{:5.1f} W/m2'.format(p['UV-A']))
    PutString(0,90,'UV-A-Index',WHITE)
    PutString(0,110,'{0}'.format(p['UV-A-Index']))

def ShowBMP280():
  "Show Pressure, Temperatur outside, and Humidity"
  title = 'BMP280'
  GetJson('weather-bmp280.json')
  SetOrientation(0)
  ClearScreen()
  SetOrientation(90)             #upside-down
  for p in data['BMP280']:
    PutString(0,0,'Luftdruck',WHITE)
    PutString(0,20,'{:4.1f} hPa'.format(p['Pressure']))
    PutString(0,45,'Temperatur',WHITE)
    PutString(0,65,'{:5.1f} deg C'.format(p['Temperature']))
    PutString(0,90,'Luftfeuchte',WHITE)
    PutString(0,110,'{:5.1f} %'.format(p['Humidity']))

def ShowBH1750():
  "Show Lightlevel"
  title = 'BH1750'
  GetJson('weather-bh1750.json')
  SetOrientation(0)
  ClearScreen()
  SetOrientation(90)             #upside-down
  for p in data['BH1750']:
    PutString(0,0,'Lightlevel',WHITE)
    PutString(0,20,'{:5.1f} lx'.format(p['Lightlevel']))

def ShowAM328P():
  "Show CurrentWindSpeed, CurrentWiindGust, CurrentWindDirection, TotalRain, IsRain"
  title = 'AM328P'
  GetJson('weather-am328p.json')
  SetOrientation(0)
  ClearScreen()
  SetOrientation(90)             #upside-down
  for p in data['AM328P']:
    PutString(0,0,'Windgeschw.',WHITE)
    PutString(0,15,'{:4.1f} m/s'.format(p['CurrentWindSpeed']))
    PutString(0,30,'max. Wind',WHITE)
    PutString(0,45,'{:4.1f} m/s'.format(p['CurrentWindGust']))
    PutString(0,60,'Windrichtung',WHITE)
    PutString(0,75,'{:5.1f} grd'.format(p['CurrentWindDirection']))
    PutString(0,90,'Regen/Regen?',WHITE)
    PutString(0,105,'{:2.5f} mm'.format(p['TotalRain']))
    PutString(120,105,'{0}'.format(p['IsRain']))

def ShowSDS011():
  "Show PM2,5, PM10"
  title = 'SDS011'
  GetJson('weather-sds011.json')
  SetOrientation(0)
  ClearScreen()
  SetOrientation(90)             #upside-down
  for p in data['SDS011']:
    PutString(0,0,'PM 2,5',WHITE)
    PutString(0,20,'{:5.1f} ppm'.format(p['pm25']))
    PutString(0,45,'PM 10',WHITE)
    PutString(0,60,'{:5.1f} ppm'.format(p['pm10']))


#########################################################################
#   Main Program
#

tWait = 5
print "Weather Display 1.0"
spi = InitSPI()                     #initialize SPI interface
InitGPIO()                          #initialize GPIO interface
InitDisplay()                       #initialize TFT controller
#RunTextTests()                      #run suite of text tests
while True:
   # Scan for cards
   (status,TagType) = MIFAREReader.MFRC522_Request(MIFAREReader.PICC_REQIDL)
   # Get the UID of the card
   (status,uid) = MIFAREReader.MFRC522_Anticoll()
   # If we have the UID, continue
#   print "Status RC522: " + str(status)
#   print "UID RC522: " + str(uid)
#   GPIO.output(LEDA, GPIO.HIGH)
   if status == MIFAREReader.MI_OK:
      # Print UID
      print("Card UID: "+str(uid[0])+","+str(uid[1])+","+str(uid[2])+","+str(uid[3]))
      if uid[0] == 183  and uid[1] == 217 and uid[2] == 123 and uid[3] == 75:
          # LED Hintergrundbeleuchtung ein (GPIO 17)
          GPIO.output(LEDA, GPIO.HIGH)

          ShowInfo()                           #Show IP, CPUTemp, Time
          time.sleep(tWait)
          ShowBMP388()                        #Show Pressure, Temperature, Altitude
          time.sleep(tWait)
          ShowMCP9808()                       #Show Temperature indoor
          time.sleep(tWait)
          ShowVEML6070()                      #Show UV Raw, UV-A, UV-A-Index
          time.sleep(tWait)
          ShowBH1750()                        #Show Lightlevel
          time.sleep(tWait)
          ShowBMP280()                        #Show Temperature, Humidity, Pressure
          time.sleep(tWait)
          ShowAM328P()                        #Show WindSpeed, WindDirection, max WIns, TotalRain, IsRain
          time.sleep(tWait)
          ShowSDS011()                        #Show PM2,5, PM10
          time.sleep(tWait)

          ClearScreen()
          # LED Hintergrundbeleuchtung aus (GPIO 17)
          GPIO.output(LEDA, GPIO.LOW)

GPIO.cleanup()
spi.close()                         #close down SPI interface
print "Done."


#   END  ###############################################################


