// ESP Gies-O-Mat MQTT Soil Moisture Controller
// Plant Soil Moisture Sense (PSMSense)
// Slave
// 
// © Copyright 2022
// Daniel Willhelm 
// Ralph Lautenschläger
//
// Arduino Config
// LOLIN (WEMOS) D1 mini
// 4M (1M LittleFS);v2 Prebuild (MSS=536); 160MHz; 
// Arduino IDE 1.8.16
// ESP8266 3.0.2
// WIFIManager 2.0.4-beta
// ArduinoJson V5
// PubSubClient 2.8.0
//
//Frontend
//Frontend open-source toolkit 3.3.7
//https://getbootstrap.com/
//
// https://randomnerdtutorials.com/esp8266-pinout-reference-gpios/
//
// D#/GPIO
// D0 - GPIO16
// D1 - GPIO5
// D2 - GPIO4
// D3 - GPIO0
// D4 - GPIO2
// D5 - GPIO14
// D6 - GPIO12
// D7 - GPIO13
// D8 - GPIO15
// RX - GPIO3
// TX - GPIO1
// A0 - ADC0
//
// ToDo
// - Values via JSON
// - input check
// - NTP timezone selection
// - adjust time if NTP Server not reachable  
//
// Bugs
// - WLAN reconnect partly not working in Battery Mode (system stay in wlan autoconfig mode)
// - max deep_sleep() time 35min - change uint64_t in ESP.cpp, ESP.h in uint32_t
//
// 0.1.0 First stable working version (@160MHz)
// 0.1.1 MQTT is working
// 0.1.2 Battery Mode
// 0.1.3 NTP
// 0.1.4 Added configurable measure interval
// 0.1.5 Changed position after decimal point to two
// 0.1.6 Added Multiplexer support for max 8 Sensors with CD74AC151
// 0.1.7 Added Sensor enable output mode setting
// 0.1.8 Support for analog sensors added (different Multiplexer needed!)
// 0.1.9 Changed direct function call via ticker to flag based via loop
// 0.1.9.1 Disable MQTT if MQTT-Host is Empty, Add ISR Option "ICACHE_RAM_ATTR", remove Multiplexer mode
// 0.1.10 Added NTP timezone
// 0.1.11 Added  measure battery voltage 
// 0.1.12 Change Number of Sensor to Sensornumber 
// 0.1.13 Added meta no-cache in buildHTMLHeader()
// 0.1.13.1 Change measure battery voltage
// 0.1.14 Delete sensor type analog
// 1.15.1 Added buildnumber autoincrement
// 1.16.2 Added self OTA update
// 1.16.5 Added OTA-Update Configuration to configuration-page
// 1.17.22 Added wakeup via MQTT
// 1.18.31 Added "I am awake" topic (Set if received wakeup command end reset if press gosleep button)
// 1.18.40 Change deep_sleep() type in ESP.h,ESP.cpp from uint64_t in uint32_t to use max deep_sleep time of 71 minutes
//         (http://www.edaboard.de/esp-deep-sleep-bug-t24751.html) 
// 1.19.46 Add RTC Memory Access to extend deep_sleep time over 71 minutes
// 1.19.73 Add Topic ".../last_transmission/sensor
// 1.19.82 Move call procedure setCallback(callbackMQTT) befor first subscribe topic
// 1.19.88 Init rtcMem.count with 0 on power mode
// 1.19.97 Reset rtcMem after replace battery (restart.reason "Power On")
// 1.20.103 Added temperature sensor DS18B20
// 1.20.108 Remove array of freqValue and battValue
// 1.20.113 Added MQTT-Server status on footer
// 1.21.136 Added subfunction for sendMQTT() call
// 1.21.138 Added ntp_sync flag
// 1.21.159 Reset wakeup counter if battery change or (NEW!) reset button pressed
// 1.22.161 Change ESPcore to 3.0.2
// 1.22.162 Change FS.h to LitleFS.h -> SPIFFS to LittleFS
// 1.22.201 Move subscribe to reconnectMQTT()
// 1.22.205 Optimize combination of client.connected() and reconnetcMQTT()
 
//
// **************************************************************************
//  Includes
// **************************************************************************
#include "version.h"
#include <ESP8266WiFi.h>
#include <DNSServer.h>                  // Local DNS Server used for redirecting all requests to the configuration portal
#include <ESP8266WebServer.h>           // Local WebServer used to serve the configuration portal
#include <ESP8266HTTPClient.h>          // HTTP Client for Self-OTA-Update
#include <ESP8266httpUpdate.h>          // Used for Self-OTA-Update
#include <WiFiManager.h>                // https://github.com/tzapu/WiFiManager WiFi Configuration Magic
#include <ESP8266HTTPUpdateServer.h>
#include <Ticker.h>                     // Periodically call of a function
#include <ArduinoJson.h>                // Config file handling
#include <TimeLib.h>                    // https://github.com/PaulStoffregen/Time
#include <Timezone.h>                   // https://github.com/JChristensen/Timezone
#include "LittleFS.h"                   // use LittleFS 
#include <OneWire.h>                    // read DS1820
#include <DallasTemperature.h>          // read DS1820
 
extern "C" {
#include "user_interface.h"             // Used for RTC Memory access
}

#define MQTT_VERSION MQTT_VERSION_3_1   // That's needed to change in PubSubClient.h !!!
#include <PubSubClient.h>               // MQTT


ESP8266WebServer httpServer(80);
ESP8266HTTPUpdateServer httpUpdater;

Ticker startMeasure;

WiFiClient espClient;                   // Only needed for MQTT PubSubClient!
PubSubClient mqttClient(espClient);     // Enable MQTT

WiFiUDP Udp;                            // Needed fpr NTP
unsigned int NtpLocalPort = 123;        // local port to listen for UDP packets (NTP)
  
// **************************************************************************
//  Defines (Do not use D3 and D8)
// **************************************************************************
#define DEBUG                         // Define for Debug output on serial interface      
#define DS18B20_SENSOR                // Define for use temperature mesurement with DS18B20
#define SENSOR_ENABLE           D1    // Gies-o-mat power supply (GPIO5)(you can connect VCC of the Sensore direcly to this pin if <20mA!)
#define SENSOR_IN               D7    // Gies-o-mat sensor input (GPIO13)
#define ONE_WIRE_BUS            D8    // Define D8 (GPIO 15) for 1-Wire-BUS-Pin
#define CLEAR_BTN               D2    // LOW during PowerUp will clear the json config file
#define POWER_MODE              D5    // High for Power Supply, Low for Battery Mode

// **************************************************************************
// Use RTC for extend deep_sleep_Counter
// **************************************************************************
#define RTCMEMORYSTART          65    // First RTC Memoryposition
typedef struct {
  int count;
} rtcStore;
rtcStore rtcMem;
//****************************************************************************

const String FIRMWARE_NAME = "PSM-Sense-Slave";
const String VERSION       = SKETCH_VERSION;
//const char* fwUrlBase = "http://192.168.XXX.XXX/PSMSenseSlave/";


// *************************************************************************
// SendMQTT Defines
// *************************************************************************
#define MQTT_LASTTRANSMISSION               // sendMQTT()
#define MQTT_ALL                      0xFF  // not in use yet
#define MQTT_TRESHOLD                 0x01  // Handle_sensor_config()
#define MQTT_MOISTURE                 0x02  // measureFreq()
#define MQTT_DS18B20                  0x04  // measureFreq()
#define MQTT_VERSION                  0x08  // checkForUpdates()
#define MQTT_BATTERY                  0x10  // measureBatt() 
#define MQTT_WAKEUP                   0x20  // Handle_GoSleep()
#define MQTT_IMAWAKE                  0x40  // callbackMQTT()
#define MQTT_ERROR                    0x80  // not in use yet

// **************************************************************************
//  Debug
// **************************************************************************
#ifdef DEBUG
#define DEBUG_PRINT(x)  Serial.print (x)
#define DEBUG_PRINTLN(x)  Serial.println (x)
#else
#define DEBUG_PRINT(x)
#define DEBUG_PRINTLN(x)
#endif

// **************************************************************************
//  Variables
// **************************************************************************
// config.json
char host_name[20] = "";
char ntpserver[30] = "";
char otau_server[30] = "";                          // Server for OTA-Updates
char otau_path[30] = "/PSMSenseSlave/";             // Path to Updatefile
bool otau_enabled = false;
char mqtt_server[30] = "";
char mqtt_topic[30] = "";
bool mqtt_enabled = false;
char sensornumber[3] = "01";
int millisecs = 0;                                   // milisecs by programmstart                                                 
int mqtt_receive_wait = 5000;      
bool gosleep = false;                                // True if call /gosleep or press Button "GoSleeep" on WebPage
char defaultmqtttopic[30] = "PSMSense/moisture/sensor/00";
int power_sampling_interval = 10;                    // Measure Frequency in Power Mode in Minutes
int bat_sampling_interval = 10;                      // Measure Frequency in Battery Mode in Minutes
bool sens_enable_mode = true;                        // Sensor enable PIN High or Low active (True = High Active)
float adc_shift=0.00;                                // Battery voltage correction

//NTP
bool ntp_enabled = false;                            // Set to false to disable querying for the time
bool ntp_sync = false;                               // if getNTPTime() received time
char poolServerName[30] = "europe.pool.ntp.org";     // default NTP Server when not configured in config.json
char boottime[20] = "";                              // PSMSene Boot Time
const int timeZone = 1;                              // Central European Time
const int NTP_PACKET_SIZE = 48;                      // NTP time is in the first 48 bytes of message
byte packetBuffer[NTP_PACKET_SIZE];                  // Buffer to hold incoming & outgoing packets

//Timezone
//Central European Time (Frankfurt, Paris)
TimeChangeRule CEST = { "CEST", Last, Sun, Mar, 2, 120 };     //Central European Summer Time
TimeChangeRule CET = { "CET ", Last, Sun, Oct, 3, 60 };       //Central European Standard Time
Timezone CE(CEST, CET);
TimeChangeRule *tcr;        //pointer to the time change rule, use to get the TZ abbrev
time_t utc, local;

// Webserver
String javaScript;
String htmlHeader;
String htmlFooter;

// Interrupt
volatile unsigned long interruptCounter = 0;    // volatile because of ISR

// Other
bool shouldSaveConfig = false;                  // Flag for saving data
float freqValue = 0;                            // Sensor Frequency
String analogBattValue = "";                    // Battery Voltage 
bool powermode = true;                          // Power Mode True for Power Supply, False for Battery Mode
bool func_measure_call = false;                 // Flag for jump into the measure function
int serialspeed = 74880;                        // Baud rate for Serial.print (boot messages are send via 74880 baud)
bool wakeup = false;                            // Set/Reset wakeup flag via mqtt for configuration
bool iamawake = false;                          // Set/Reset iamawake flag if ESP received wakeup command

#ifdef DS18B20_SENSOR
  //Temperature
  OneWire oneWire(ONE_WIRE_BUS);
  DallasTemperature DS18B20(&oneWire);
  float temperature;                              // store temperature
#endif  

// **************************************************************************
// JSON Configuration Management
// **************************************************************************
bool loadConfig() {
  if (LittleFS.exists("/config.json")) {
      File configFile = LittleFS.open("/config.json", "r");
      if (configFile) {
        DEBUG_PRINTLN("opened config file");
        size_t size = configFile.size();
        if (size > 1024) {
          DEBUG_PRINTLN("Config file size is too large");
          return false;
        }
        std::unique_ptr<char[]> buf(new char[size]);            // Allocate a buffer to store contents of the file.
        configFile.readBytes(buf.get(), size);                  // Input Buffer (needed by ArduinoJson)
        DynamicJsonBuffer jsonBuffer;
        JsonObject& json = jsonBuffer.parseObject(buf.get());
        #ifdef DEBUG
          json.printTo(Serial);
        #endif
        if (json.success()) {
          DEBUG_PRINTLN("\nparsed json");
          if (json.containsKey("hostname")) strncpy(host_name, json["hostname"], 20);
          if (json.containsKey("mqttserver")) strncpy(mqtt_server, json["mqttserver"], 30);
          if (json.containsKey("mqtttopic")) strncpy(mqtt_topic, json["mqtttopic"], 30);
          if (json.containsKey("mqttenabled")) mqtt_enabled = json["mqttenabled"];
          if (json.containsKey("ntpserver")) strncpy(ntpserver, json["ntpserver"], 30);
          if (json.containsKey("ntpenabled")) ntp_enabled = json["ntpenabled"];
          if (json.containsKey("otauserver")) strncpy(otau_server, json["otauserver"], 30);
          if (json.containsKey("otauenabled")) otau_enabled = json["otauenabled"];
          if (json.containsKey("sampleintpower")) power_sampling_interval = json["sampleintpower"];
          if (json.containsKey("sampleintbat")) bat_sampling_interval = json["sampleintbat"];
          if (json.containsKey("sensornumber")) strncpy(sensornumber , json["sensornumber"], 3);
          if (json.containsKey("sensorsenablemode")) sens_enable_mode = json["sensorsenablemode"];
          if (json.containsKey("adc_shift")) adc_shift = json["adc_shift"];
          if (mqtt_server == "") mqtt_enabled = 0;

          DEBUG_PRINT("loadConfig: Read json[otauenabled]: ");
          DEBUG_PRINTLN(otau_enabled);
        
        }
        else {
          DEBUG_PRINTLN("Failed to parse config file");
          return false;
        }
      }
      else {
        DEBUG_PRINTLN("Failed to open config file");
        return false;
      }
  }
  return true;
}


// **************************************************************************
//  Callback notifying us of the need to save config
// **************************************************************************
void saveConfigCallback ()
{
  DEBUG_PRINTLN("Should save config");
  shouldSaveConfig = true;
}


// **************************************************************************
//  Gets called when WiFiManager enters configuration mode
// **************************************************************************
void configModeCallback (WiFiManager *myWiFiManager)
{
  DEBUG_PRINTLN("Entered config mode");
  DEBUG_PRINTLN(WiFi.softAPIP());
  DEBUG_PRINTLN(myWiFiManager->getConfigPortalSSID());    //if you used auto generated SSID, print it
}


// **************************************************************************
//  IP Address to String
// **************************************************************************
String ipToString(IPAddress ip)
{
  String s = "";
  for (int i = 0; i < 4; i++)
    s += i  ? "." + String(ip[i]) : String(ip[i]);
  return s;
}


// **************************************************************************
//  NTP Time Syncronization
// **************************************************************************
time_t getNtpTime()
{
  IPAddress ntpServerIP;                            // NTP server's ip address

  while (Udp.parsePacket() > 0) ;                   // Discard any previously received packets
  DEBUG_PRINTLN("NTP: Transmit NTP Request");
  WiFi.hostByName(ntpserver, ntpServerIP);          // Lookup IP from Hostname
  DEBUG_PRINT("NTP: ");
  DEBUG_PRINT(ntpserver);
  DEBUG_PRINT(" IP: ");
  DEBUG_PRINTLN(ntpServerIP);
  sendNTPpacket(ntpServerIP);
  uint32_t beginWait = millis();
  while (millis() - beginWait < 1500) {
    int size = Udp.parsePacket();
    if (size >= NTP_PACKET_SIZE) {
      DEBUG_PRINTLN("NTP: Receive NTP Response");
      Udp.read(packetBuffer, NTP_PACKET_SIZE);  // read packet into the buffer
      unsigned long secsSince1900;
      // convert four bytes starting at location 40 to a long integer
      secsSince1900 =  (unsigned long)packetBuffer[40] << 24;
      secsSince1900 |= (unsigned long)packetBuffer[41] << 16;
      secsSince1900 |= (unsigned long)packetBuffer[42] << 8;
      secsSince1900 |= (unsigned long)packetBuffer[43];
      secsSince1900 -= 2208988800UL;  //+ (timeZone * 60 * 60); // NTP Time - 70 Years + TimeZone Hours
      ntp_sync = true;
      return CE.toLocal(secsSince1900, &tcr);
    }
  }
  DEBUG_PRINTLN("NTP: No NTP Response :-(");
  ntp_sync = false;  
  return 0;                                          // return 0 if unable to get the time
}

// send an NTP request to the time server at the given address
void sendNTPpacket(IPAddress &address)
{
  memset(packetBuffer, 0, NTP_PACKET_SIZE);                 // set all bytes in the buffer to 0
  packetBuffer[0] = 0b11100011;                             // LI (Clock is unsynchronized), Version (4), Mode (Client)
  packetBuffer[1] = 0;                                      // Stratum, or type of clock (Unspecified or invalid)
  packetBuffer[2] = 6;                                      // Polling Interval (log2 seconds)
  packetBuffer[3] = 0xEC;                                   // Peer Clock Precision (log2 seconds)
                                                            // 8 bytes of zero for Root Delay & Root Dispersion
  packetBuffer[12] = 49;                                    // 
  packetBuffer[13] = 0x4E;                                  //
  packetBuffer[14] = 49;                                    //
  packetBuffer[15] = 52;                                    //
  Udp.beginPacket(address, 123);                            // NTP requests are to port 123
  Udp.write(packetBuffer, NTP_PACKET_SIZE);
  Udp.endPacket();
}


// **************************************************************************
//  HTML Header Template
// **************************************************************************
void buildHeader()
{
  htmlHeader="<!DOCTYPE html PUBLIC '-//W3C//DTD XHTML 1.0 Strict//EN' 'http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd'>\n";
  htmlHeader+="<html xmlns='http://www.w3.org/1999/xhtml' xml:lang='en'>\n";
  htmlHeader+="  <head>\n";
  htmlHeader+="    <meta name='viewport' content='width=device-width, initial-scale=.75' />\n";
  htmlHeader+="    <meta http-equiv='Cache-Control' content='no-cache, no-store, must-revalidate' />\n";
  htmlHeader+="    <meta http-equiv='Pragma' content='no-cache' />\n";
  htmlHeader+="    <meta http-equiv='Expires' content='0' />\n";
  htmlHeader+="    <link rel='stylesheet' href='https://maxcdn.bootstrapcdn.com/bootstrap/3.3.7/css/bootstrap.min.css' />\n";
  htmlHeader+="    <style>@media (max-width: 991px) {.nav-pills>li {float: none; margin-left: 0; margin-top: 5px; text-align: center;}}</style>\n";
  htmlHeader+="    <title>" + FIRMWARE_NAME + " - V " + VERSION + " - Sensor " + sensornumber + "</title>\n";
  htmlHeader+="  </head>\n";
  htmlHeader+="  <body>\n";
  htmlHeader+="    <div class='container'>\n";
  htmlHeader+="      <h1>" + FIRMWARE_NAME + " - V " + VERSION + " - Sensor " + sensornumber + "</h1>\n";
  htmlHeader+="      <div class='row'>\n";
  htmlHeader+="        <div class='col-md-12'>\n";
  htmlHeader+="          <ul class='nav nav-pills'>\n";
  htmlHeader+="            <li class='active'>\n";
  htmlHeader+="              <a href='#'>Hostname <span class='badge'>" + String(host_name) + "</span></a></li>\n";
  htmlHeader+="            <li class='active'>\n";
  htmlHeader+="              <a href='http://" + ipToString(WiFi.localIP()) + ":80'>Local IP<span class='badge'>" + ipToString(WiFi.localIP()) + "</span></a></li>\n";
  htmlHeader+="            <li class='active'>\n";
  htmlHeader+="              <a href='#'>MAC <span class='badge'>" + String(WiFi.macAddress()) + "</span></a></li>\n";
  htmlHeader+="            <li class='active'>\n";
  htmlHeader+="              <a href='/config'>Configuration</a></li>\n";
  htmlHeader+="            <li class='active'>\n";
  htmlHeader+="              <a href='#'><span class='glyphicon glyphicon-signal'></span> "+ String(WiFi.RSSI()) + " dBm</a></li>\n";
  htmlHeader+="          </ul>\n";
  htmlHeader+="        </div>\n";
  htmlHeader+="      </div><hr />\n";
}


// **************************************************************************
//  HTML Footer Template
// **************************************************************************
void buildFooter()
{
  char nowTime[10];
  
  sprintf(nowTime, "%02d:%02d:%02d", hour(now()), minute(now()), second(now()));
  htmlFooter="       <br><br><hr />";
  htmlFooter+="      <div class='row'><div class='col-md-12'><em>MQTT Plant Soil Moisture Sense Gies-O-Mat -- V " + VERSION + " -- Zeit: "+ nowTime +"</em></div></div>\n";
  if (otau_enabled) {
      htmlFooter+="      <div class='row'><div class='col-md-12'><em>OTA-Update Server: http://" + String(otau_server) + String(otau_path) + "</em></div></div>\n";
  } else {
      htmlFooter+="      <div class='row'><div class='col-md-12'><em>OTA-Update: disabled </em></div></div>\n";
  }
  if ((mqtt_enabled) && (mqttClient.connected())){
      htmlFooter+="      <div class='row'><div class='col-md-12'><em>MQTT Server: " + String(mqtt_server) + "</em></div></div>\n";    
  } else if ((mqtt_enabled) && (!mqttClient.connected())){
      htmlFooter+="      <div class='row'><div class='col-md-12'><em>MQTT Server: " + String(mqtt_server) + " not available</em></div></div>\n";    
  } else if (!mqtt_enabled){
      htmlFooter+="      <div class='row'><div class='col-md-12'><em>MQTT Server: disabled</em></div></div>\n";    
  }
  
  htmlFooter+="    </div>\n";
  htmlFooter+="  </body>\n";
  htmlFooter+="</html>\n";
}


// **************************************************************************
//  HTML Page for ESP Reboot
// **************************************************************************
void Handle_Reboot()
{
  httpServer.sendHeader("Connection", "close");
  httpServer.send(200, "text/html", F("<body>Reboot OK, please reload page.</body>"));
  delay(500);
  if (bat_sampling_interval >= 60) {
     rtcMem.count++;                         // Restart reduced counter by 1
     writeToRTCMemory();                     
  }  

  ESP.restart();
}

// **************************************************************************
//  HTML Page for ESP Reboot
// **************************************************************************
void Handle_GoSleep()
{
  httpServer.sendHeader("Connection", "close");
  httpServer.send(200, "text/html", F("<body>Go sleeping. Good night!</body>"));
  delay(500);
  gosleep = true;  
  sendMQTT(MQTT_WAKEUP);   // Reset wakeup topic
  delay(500);
  ESP.restart();
}

// **************************************************************************
//  HTML Page for ESP Clear Json config-file
// **************************************************************************
void Handle_ClearConfig()
{
  httpServer.sendHeader("Connection", "close");
  httpServer.send(200, "text/html", F("<body>Config File deleted, ESP reboot, please reload page.</body>"));
  LittleFS.remove("/config.json");
  delay(500);
  ESP.restart();
}

// **************************************************************************
//  HTML Page for ESP store Json config-file
// **************************************************************************
void Handle_config()
{
  if (httpServer.method() == HTTP_GET) {
    DEBUG_PRINTLN("WEB: Connection received - /config");
    sendConfigPage("", "", 0, 200);
  } else {
    DEBUG_PRINTLN("WEB: Connection received - /config (save)");
    sendConfigPage("Settings saved successfully! Please Reboot!", "Success!", 1, 200);
    }
}

// **************************************************************************
//  HTML Send config-Page
// **************************************************************************
void sendConfigPage(String message, String header, int type, int httpcode)
{
  char host_name_conf[20] = "";
  char mqtt_server_conf[30] = "";
  char mqtt_topic_conf[30] = "";
  char otau_server_conf[30] = "";
  char ntpserver_conf[30] = "";
  bool ntpenabled_conf;
  bool mqttenabled_conf;
  bool otauenabled_conf;
  int power_sampling_interval_conf;
  int bat_sampling_interval_conf;
  char sensor_number_conf[3] ="";
  bool sens_enable_mode_conf;
  float adc_shift_conf;

  if (type == 1){                                              // Type 1 -> save data
    String message = "WEB: Number of args received:";
    message += String(httpServer.args()) + "\n";
    for (int i = 0; i < httpServer.args(); i++) {
      message += "Arg " + (String)i + " ==> ";
      message += httpServer.argName(i) + ":" ;
      message += httpServer.arg(i) + "\n";
    }
    if (httpServer.hasArg("ntpenabled")) {ntpenabled_conf = true;} else {ntpenabled_conf = false;}
    if (httpServer.hasArg("mqttenabled")) {mqttenabled_conf = true;} else {mqttenabled_conf = false;}
    if (httpServer.hasArg("otauenabled")) {otauenabled_conf = true;} else {otauenabled_conf = false;}
    strncpy(host_name_conf, httpServer.arg("host_name_conf").c_str(), 20);
    strncpy(mqtt_server_conf, httpServer.arg("mqtt_server_conf").c_str(), 30);
    strncpy(mqtt_topic_conf, httpServer.arg("mqtt_topic_conf").c_str(), 30);
    strncpy(ntpserver_conf, httpServer.arg("ntpserver_conf").c_str(), 30);
    strncpy(otau_server_conf, httpServer.arg("otau_server_conf").c_str(), 30);    
    power_sampling_interval_conf = atoi(httpServer.arg("power_sampling_interval_conf").c_str());
    bat_sampling_interval_conf = atoi(httpServer.arg("bat_sampling_interval_conf").c_str());
    strncpy(sensor_number_conf, httpServer.arg("sensor_number_conf").c_str(), 3);
    if (httpServer.arg("sensorsenablemode") == "High") {sens_enable_mode_conf = true;} else {sens_enable_mode_conf = false;}
    adc_shift_conf = atof(httpServer.arg("adc_shift_conf").c_str());
    adc_shift = adc_shift_conf;                                                                 //Update adc_shift for main-page and battery measurement
    otau_enabled = otauenabled_conf;
    strncpy(otau_server,otau_server_conf,30);     
    DEBUG_PRINTLN(message);

    // validate values before saving
    bool validconf = true;
    if (validconf)
    {
      DEBUG_PRINTLN("SPI: save config.json...");
      DynamicJsonBuffer jsonBuffer;
      JsonObject& json = jsonBuffer.createObject();
      json["hostname"] = String(host_name_conf);
      json["mqttserver"] = String(mqtt_server_conf);
      json["mqtttopic"] = String(mqtt_topic_conf);
      json["mqttenabled"] = String(mqttenabled_conf);
      json["ntpserver"] = String(ntpserver_conf);
      json["ntpenabled"] = String(ntpenabled_conf);
      json["otauserver"] = String(otau_server_conf);
      json["otauenabled"] = String(otauenabled_conf);
      json["sampleintpower"] = String(power_sampling_interval_conf);
      json["sampleintbat"] = String(bat_sampling_interval_conf);
      json["sensornumber"] = String(sensor_number_conf);
      json["sensorsenablemode"] = String(sens_enable_mode_conf);
      json["adc_shift"] = String(adc_shift_conf);

      File configFile = LittleFS.open("/config.json", "w");
      if (!configFile) {
        DEBUG_PRINTLN("SPI: failed to open config file for writing");
      }
      #ifdef DEBUG
        json.printTo(Serial);
      #endif
      DEBUG_PRINTLN("");
      json.printTo(configFile);
      configFile.close();
      //end save
      
      // Change Deep-Sleep-Counter in RTCMemory
      bat_sampling_interval = bat_sampling_interval_conf;
      if (bat_sampling_interval >= 60) {
         rtcMem.count = bat_sampling_interval/60;                         // Load counter for full houres
         DEBUG_PRINTLN("Writing wakup counter to RTCMemory");
         writeToRTCMemory();                                              // New value after change config
      } else {   
         rtcMem.count = 0;                                                // Clear rtcMem.count
         DEBUG_PRINTLN("bat_sampling_interval < 60 -> Clear wakup counter in RTCMemory");
         writeToRTCMemory();                                              // New value after change config
      }  

    }
  } else {                                // Type 0 -> load data
    if (LittleFS.begin())
    {
      DEBUG_PRINTLN("SPI: mounted file system");
      if (LittleFS.exists("/config.json"))
      {
        //file exists, reading and loading
        DEBUG_PRINTLN("SPI: reading config file");
        File configFile = LittleFS.open("/config.json", "r");
        if (configFile) {
          DEBUG_PRINTLN("SPI: opened config file");
          size_t size = configFile.size();
          // Allocate a buffer to store contents of the file.
          std::unique_ptr<char[]> buf(new char[size]);

          configFile.readBytes(buf.get(), size);
          DynamicJsonBuffer jsonBuffer;
          JsonObject& json = jsonBuffer.parseObject(buf.get());
          DEBUG_PRINT("JSO: ");
          #ifdef DEBUG
            json.printTo(Serial);
          #endif
          if (json.success()) {
            DEBUG_PRINTLN("\nJSO: parsed json");
            if (json.containsKey("hostname")) strncpy(host_name_conf, json["hostname"], 20);
            if (json.containsKey("mqttserver")) strncpy(mqtt_server_conf, json["mqttserver"], 30);
            if (json.containsKey("mqtttopic")) strncpy(mqtt_topic_conf, json["mqtttopic"], 30);
            if (json.containsKey("mqttenabled")) mqttenabled_conf = json["mqttenabled"];
            if (json.containsKey("ntpserver")) strncpy(ntpserver_conf, json["ntpserver"], 30);
            if (json.containsKey("ntpenabled")) ntpenabled_conf = json["ntpenabled"];
            if (json.containsKey("otauserver")) strncpy(otau_server_conf, json["otauserver"], 30);
            if (json.containsKey("otauenabled")) otauenabled_conf = json["otauenabled"];
            if (json.containsKey("sampleintpower")) power_sampling_interval_conf = json["sampleintpower"];
            if (json.containsKey("sampleintbat")) bat_sampling_interval_conf = json["sampleintbat"];
            if (json.containsKey("sensornumber")) strncpy(sensor_number_conf, json["sensornumber"], 3);
            if (json.containsKey("sensorsenablemode")) sens_enable_mode_conf = json["sensorsenablemode"];
            if (json.containsKey("adc_shift")) adc_shift_conf = json["adc_shift"];
          } else {
            DEBUG_PRINTLN("JSO: failed to load json config");
          }
        }
      }
    } else {
      DEBUG_PRINTLN("SPI: failed to mount FS");
    }
  }
  String htmlDataconf;        // Hold the HTML Code
  buildHeader();
  buildFooter();
  htmlDataconf=htmlHeader;
  
  if (type == 1)
    htmlDataconf+="      <div class='row'><div class='col-md-12'><div class='alert alert-success'><strong>" + header + "!</strong> " + message + "</div></div></div>\n";
  if (type == 2)
    htmlDataconf+="      <div class='row'><div class='col-md-12'><div class='alert alert-warning'><strong>" + header + "!</strong> " + message + "</div></div></div>\n";
  if (type == 3)
    htmlDataconf+="      <div class='row'><div class='col-md-12'><div class='alert alert-danger'><strong>" + header + "!</strong> " + message + "</div></div></div>\n";
  htmlDataconf+="      <div class='row'>\n";
  htmlDataconf+="<form method='post' action='/config'>";
  htmlDataconf+="        <div class='col-md-12'>\n";
  htmlDataconf+="          <h3>Configuration</h3>\n";
  htmlDataconf+="          <table class='table table-striped' style='table-layout: fixed;'>\n";
  htmlDataconf+="            <thead><tr><th>Option</th><th>Current Value</th><th>New Value</th></tr></thead>\n";
  htmlDataconf+="            <tbody>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>Hostname</td><td><code>" + ((host_name_conf[0] == 0 ) ? String("(" + String(host_name) + ")") : String(host_name_conf)) + "</code></td><td><input type='text' id='host_name_conf' name='host_name_conf' value='" + String(host_name_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>MQTT Server</td><td><code>" + String(mqtt_server_conf) + "</code></td><td><input type='text' id='mqtt_server_conf' name='mqtt_server_conf' value='" + String(mqtt_server_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>MQTT Topic</td><td><code>" + String(mqtt_topic_conf) + "</code></td><td><input type='text' id='mqtt_topic_conf' name='mqtt_topic_conf' value='" + String(mqtt_topic_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>MQTT enabled?</td><td><code>" + (mqttenabled_conf ? String("Yes") : String("No")) + "</code></td><td><input type='checkbox' id='mqttena' name='mqttenabled' " + (mqttenabled_conf ? String("checked") : String("")) + "></td></tr>";
  htmlDataconf+="            <tr class='text-uppercase'><td>NTP Server</td><td><code>" + ((ntpserver_conf[0] == 0 ) ? String("(" + String(poolServerName) + ")") : String(ntpserver_conf)) + "</code></td><td><input type='text' id='ntpserver_conf' name='ntpserver_conf' value='" + String(ntpserver_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>NTP enabled?</td><td><code>" + (ntpenabled_conf ? String("Yes") : String("No")) + "</code></td><td><input type='checkbox' id='ntpena' name='ntpenabled' " + (ntpenabled_conf ? String("checked") : String("")) + "></td></tr>";
  htmlDataconf+="            <tr class='text-uppercase'><td>OTA-Update Server</td><td><code>" + String(otau_server_conf) + "</code></td><td><input type='text' id='otau_server_conf' name='otau_server_conf' value='" + String(otau_server_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>OTA-Update enabled?</td><td><code>" + (otauenabled_conf ? String("Yes") : String("No")) + "</code></td><td><input type='checkbox' id='otauena' name='otauenabled' " + (otauenabled_conf ? String("checked") : String("")) + "></td></tr>";
  htmlDataconf+="            <tr class='text-uppercase'><td>Measure Freq Power Mode (minutes)</td><td><code>" + String(power_sampling_interval_conf) + "</code></td><td><input type='text' id='power_sampling_interval_conf' name='power_sampling_interval_conf' value='" + String(power_sampling_interval_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>Measure Freq Battery Mode (minutes)</td><td><code>" + String(bat_sampling_interval_conf) + "</code></td><td><input type='text' id='bat_sampling_interval_conf' name='bat_sampling_interval_conf' value='" + String(bat_sampling_interval_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>Number of Sensor</td><td><code>" + String(sensor_number_conf) + "</code></td><td><select name='sensor_number_conf' size='2'><option " + (strncmp(sensor_number_conf,"01",2) ? String("") : String("selected")) + ">01</option><option "+ (strncmp(sensor_number_conf,"02",2) ? String("") : String("selected")) +">02</option><option "+ (strncmp(sensor_number_conf,"03",2) ? String("") : String("selected")) +">03</option><option "+ (strncmp(sensor_number_conf,"04",2) ? String("") : String("selected")) +">04</option><option "+ (strncmp(sensor_number_conf,"05",2) ? String("") : String("selected")) +">05</option><option "+ (strncmp(sensor_number_conf,"06",2) ? String("") : String("selected")) +">06</option><option "+ (strncmp(sensor_number_conf,"07",2) ? String("") : String("selected")) +">07</option><option "+ (strncmp(sensor_number_conf,"08",2) ? String("") : String("selected")) +">08</option><option "+ (strncmp(sensor_number_conf,"09",2) ? String("") : String("selected")) +">09</option><option "+ (strncmp(sensor_number_conf,"10",2) ? String("") : String("selected")) +">10</option></select></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>Sensor Enable Port</td><td><code>" + (sens_enable_mode_conf ? String("High active") : String("Low active")) + "</code></td><td><select name='sensorsenablemode' size='1'><option " + ((sens_enable_mode_conf ) ? String("selected") : String("")) + ">High</option><option "+ ((!sens_enable_mode_conf ) ? String("selected") : String("")) +">Low</option></select></td></tr>";
  htmlDataconf+="            <tr class='text-uppercase'><td>Battery voltage correction</td><td><code>" + String(adc_shift_conf) + "</code></td><td><input type='text' id='adc_shift_conf' name='adc_shift_conf' value='" + String(adc_shift_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr><td colspan='5' class='text-center'><em><a href='/reboot' class='btn btn-sm btn-danger'>Reboot</a>  <a href='/update' class='btn btn-sm btn-warning'>Update</a>  <button type='submit' class='btn btn-sm btn-success'>Save</button>  <a href='/' class='btn btn-sm btn-primary'>Cancel</a>";
  if (wakeup)
    htmlDataconf+="          <a href='/gosleep' class='btn btn-sm btn-light'>Go sleeping</a>";  
  htmlDataconf+="            </em></td></tr>";
  htmlDataconf+="            </tbody></table>\n";
  htmlDataconf+="          </div></div>\n";
  
  htmlDataconf+=htmlFooter;

  httpServer.send(httpcode, "text/html; charset=utf-8", htmlDataconf);
  httpServer.client().stop();
}

// **************************************************************************
//  HTML get Value Page
// **************************************************************************
void Handle_getvalue(){
  String htmlDataconf;        // Hold the HTML Code
  buildHeader();
  buildFooter();
  htmlDataconf=htmlHeader;
  
  htmlDataconf+="Battery value messure : " + String(analogBattValue) + " V<br><br>";
#ifdef DS18B20_SENSOR
  htmlDataconf+="Temperature value messure : " + String(temperature) + " °C<br><br>";
#endif
  htmlDataconf+="Last value messure : " + String(freqValue) + " kHz<br><br>";
    
  htmlDataconf+=htmlFooter;
    
  httpServer.send(200, "text/html; charset=utf-8", htmlDataconf);
  httpServer.client().stop(); 
}

// **************************************************************************
//  HTML Main Page
// **************************************************************************
void Handle_welcome(){
  
  String htmlDataMain;        // Hold the HTML Code
  buildHeader();
  buildFooter();
  htmlDataMain=htmlHeader;
  
  htmlDataMain+="        <div class='col-md-12'>\n";
  htmlDataMain+="          <h3>Plant Soil Moisture Sense</h3><br>";
  htmlDataMain+="Last value messure : " + String(freqValue) + " kHz<br><br>";
  htmlDataMain+="Battery value messure : " + String(analogBattValue) + " V<br><br>";
#ifdef DS18B20_SENSOR
  htmlDataMain+="Temperature value messure : " + String(temperature) + " °C<br><br>";
#endif
  htmlDataMain+="           Possible URL Prefix:<br>";
  htmlDataMain+="           /reboot<br>";
  htmlDataMain+="           /gosleep<br>";
  htmlDataMain+="           /clearconfig<br>";
  htmlDataMain+="           /freemem<br>";
  htmlDataMain+="           /getvalue<br>";
  htmlDataMain+="           /config<br><br><br>";
  htmlDataMain+="          </div><br>";
  
  htmlDataMain+=htmlFooter;

  httpServer.send(200, "text/html; charset=utf-8", htmlDataMain);
  httpServer.client().stop();  
}

// **************************************************************************
//  MQTT reconnect & subscribe
// **************************************************************************
bool reconnectMQTT() {
  char mqtt_wakeup_topic[40] = "";
  boolean rc = false;
  int i=5;                                                          // Try 5 sec to connect to broker 

  while ((!mqttClient.connected())&&(i > 0)) {                      // Wait for MQTT-Connection
     DEBUG_PRINT("RECONNECT MQTT: Attempting MQTT connection...");
     if (mqttClient.connect(host_name)) {
         DEBUG_PRINTLN("connected");
         strcat (mqtt_wakeup_topic,mqtt_topic);
         strcat (mqtt_wakeup_topic,"/wakeup/sensor/");              // Own topic for wakeup
         strcat (mqtt_wakeup_topic,sensornumber);
         DEBUG_PRINT("RECONNECT MQTT: Wakup-Topic: ");
         DEBUG_PRINTLN(mqtt_wakeup_topic);

         // Specify the QoS to subscribe at. Only supports QoS 0 or 1:
         rc = mqttClient.subscribe(mqtt_wakeup_topic,1);            // Subscribe wakeup flag
         if (!rc) {
            DEBUG_PRINTLN("RECONNECT MQTT: wakeup state subscribe failed ...");
         } else {
            DEBUG_PRINTLN("RECONNECT MQTT: wakeup state subscribe successful ...");
         }
         mqtt_wakeup_topic[0] = '\0';                          // Empty the array that strcat can work again
         return true;
      } else {
         i--;
         delay(1000);
         DEBUG_PRINTLN(".try again");
      }  
   }
   if (i==0) {
     DEBUG_PRINTLN("RECONNECT MQTT: Not connected to broker");
     return false;
   }
   return true;
}

// **************************************************************************
//  MQTT Recieve subscribing Frequency and Battery Value from MQTT Broker
// **************************************************************************
// C substring function definition
void substring(char s[], char sub[], int p, int l) {
   int c = 0;
   
   while (c < l) {
      sub[c] = s[p+c-1];
      c++;
   }
   sub[c] = '\0';
}

void callbackMQTT(char* topic, byte* payload, unsigned int length) {
  const char cbWakeup[] = "wakeup";
  
  // Hilfsvariablen für die Konvertierung der Nachricht in einen String
  char message_buff[100]; 
  char sensornr[3];
  char *ptr;
  int i;

  DEBUG_PRINT("Message arrived [");
  DEBUG_PRINT(topic);
  DEBUG_PRINT("] ");
  for (i=0;i<length;i++) {
    DEBUG_PRINT((char)payload[i]);
    message_buff[i] =(char)payload[i];
  }
  message_buff[i+1]='\0';
  DEBUG_PRINTLN();

  // kopiere 2 letzte Zeichen von topic (Sensornummer)
  substring(topic, sensornr, strlen(topic)-1, 2);
  
  DEBUG_PRINT("Sensornr: ");
  DEBUG_PRINTLN(sensornr);

  // Ende-Zeichen setzen
  sensornr[3] = '\0';
  
  // receive wakeup topic for slave with sensornr x ?
  ptr = strstr(topic, cbWakeup);
  if ((ptr) && (strncmp(sensornr,sensornumber,2)==0) && !powermode && !wakeup)
  {
     (strncmp(message_buff, "1", 1) == 0 ) ? wakeup = true : wakeup = false;            // Master go sleep? 
     DEBUG_PRINT("MQTT: Receive wakeup flag: ");
     DEBUG_PRINTLN(wakeup);
     enableWebService();
     sendMQTT(MQTT_IMAWAKE); // send "I am awake"
  } 
}

// **************************************************************************
//  MQTT Publish Frequency to MQTT Broker
// **************************************************************************
void sendMQTT(int sendTopic) {
  bool mqttresult = false;
  char mqtt_moisture_topic[40] = "";
  char mqtt_last_transmission_topic[40] = "";
  char mqtt_battery_topic[40] = "";
  char mqtt_version_topic[40] = "";
  char mqtt_wakeup_topic[40] = "";
  char mqtt_iamawake_topic[40] = "";
#ifdef DS18B20_SENSOR
  char mqtt_temperature_topic[40] = "";
#endif
  char nowTime[40];
  
  sprintf(nowTime, "%04d-%02d-%02d %02d:%02d:%02d ", year(now()), month(now()), day(now()), hour(now()), minute(now()), second(now()));
  DEBUG_PRINT("Sensornumber: ");
  DEBUG_PRINTLN(sensornumber);

  if ((mqtt_enabled) && (reconnectMQTT())) {
    DEBUG_PRINTLN("sendMQTT: mqtt_enable=true && mqttClient.connected()=true");
      
#ifdef DS18B20_SENSOR
    // Publish Temperature Value
    if (sendTopic & MQTT_DS18B20) {
      strcat (mqtt_temperature_topic,mqtt_topic);
      strcat (mqtt_temperature_topic,"/temperature/sensor/");
      strcat (mqtt_temperature_topic,sensornumber);
      mqttresult = mqttClient.publish(mqtt_temperature_topic, String(temperature).c_str(), true);

      if (!mqttresult) {
        DEBUG_PRINTLN("MQTT temperature_topic publish failed ...");
      } else {
        DEBUG_PRINTLN("MQTT temperature_topic publish successful ...");
      }
    }  
#endif

    // Publish Battery Value
    if (sendTopic & MQTT_BATTERY) {
      strcat (mqtt_battery_topic,mqtt_topic);
      strcat (mqtt_battery_topic,"/battery/sensor/");
      strcat (mqtt_battery_topic,sensornumber);
      mqttresult = mqttClient.publish(mqtt_battery_topic, analogBattValue.c_str(), true);

      if (!mqttresult) {
        DEBUG_PRINTLN("MQTT battery_topic publish failed ...");
      } else {
        DEBUG_PRINTLN("MQTT battery_topic publish successful ...");
      }
    }
  
    // Publish Moisture Value
    if (sendTopic & MQTT_MOISTURE) {    
      strcat (mqtt_moisture_topic,mqtt_topic);
      strcat (mqtt_moisture_topic,"/moisture/sensor/");
      strcat (mqtt_moisture_topic,sensornumber);
      mqttresult = mqttClient.publish(mqtt_moisture_topic, String(freqValue).c_str(), true);

      if (!mqttresult) {
        DEBUG_PRINTLN("MQTT moisture_topic publish failed ...");
      } else {
        DEBUG_PRINTLN("MQTT moisture_topic publish successful ...");
      }
    }
  
    // Publish current version
    if (sendTopic & MQTT_VERSION) {
      strcat (mqtt_version_topic,mqtt_topic);
      strcat (mqtt_version_topic,"/firmware/version/sensor/");
      strcat (mqtt_version_topic,sensornumber);

      DEBUG_PRINT("MQTT current VERSION-Topic: "); 
      DEBUG_PRINTLN(mqtt_version_topic);
      
      mqttresult = mqttClient.publish(mqtt_version_topic, String(VERSION).c_str(), true);  

      DEBUG_PRINT("sendMQTT: MQTT_VERSION - mqttresult: "); 
      DEBUG_PRINTLN(mqttresult);

    
      if (!mqttresult) {
        DEBUG_PRINTLN("MQTT current VERSION publish failed ...");
      } else {
        DEBUG_PRINT("MQTT current VERSION ("); 
        DEBUG_PRINT(VERSION);
        DEBUG_PRINTLN(") publish successful ...");
      }
    }
    
    // Set "I am awake" flag
    if (wakeup && !gosleep && (sendTopic & MQTT_IMAWAKE)) {
      strcat (mqtt_iamawake_topic,mqtt_topic);
      strcat (mqtt_iamawake_topic,"/iamawake/sensor/");
      strcat (mqtt_iamawake_topic,sensornumber);
      mqttresult = mqttClient.publish(mqtt_iamawake_topic, String(true).c_str(), true);  
      
      if (!mqttresult) {
        DEBUG_PRINTLN("MQTT I am awake SET publish failed ...");
      } else {
        DEBUG_PRINTLN("MQTT I am awake SET publish successful ...");
      }
    }

    // Reset wakeup flag
    if (wakeup && gosleep && (sendTopic & MQTT_WAKEUP)) {
      strcat (mqtt_wakeup_topic,mqtt_topic);
      strcat (mqtt_wakeup_topic,"/wakeup/sensor/");
      strcat (mqtt_wakeup_topic,sensornumber);
      mqttresult = mqttClient.publish(mqtt_wakeup_topic, String(!wakeup).c_str(), true);  
      
      if (!mqttresult) {
        DEBUG_PRINTLN("MQTT wakeup RESET publish failed ...");
      } else {
        DEBUG_PRINTLN("MQTT wakeup RESET publish successful ...");
      }

      // Reset "I am awake" flag
      strcat (mqtt_iamawake_topic,mqtt_topic);
      strcat (mqtt_iamawake_topic,"/iamawake/sensor/");
      strcat (mqtt_iamawake_topic,sensornumber);
      mqttresult = mqttClient.publish(mqtt_iamawake_topic, String(false).c_str(), true);  
    
      if (!mqttresult) {
        DEBUG_PRINTLN("MQTT I am awake SET publish failed ...");
      } else {
        DEBUG_PRINTLN("MQTT I am awake SET publish successful ...");
      }
    }

    // Publish Last transmission Value
    // Publish time only if ntp in sync
    if (ntp_sync) {
      strcat (mqtt_last_transmission_topic,mqtt_topic);
      strcat (mqtt_last_transmission_topic,"/last-transmission/sensor/");
      strcat (mqtt_last_transmission_topic,sensornumber);
      mqttresult = mqttClient.publish(mqtt_last_transmission_topic, nowTime, true);

      if (!mqttresult) {
        DEBUG_PRINTLN("MQTT mqtt_last_transmission_topic publish failed ...");
      } else {
        DEBUG_PRINTLN("MQTT mqtt_last_transmission_topic publish successful ...");
      }
    } else {
        DEBUG_PRINTLN("MQTT mqtt_last_transmission_topic not publish - ntp not in sync.");
    }
    
  } else {
     DEBUG_PRINTLN("sendMQTT: MQTT publish failed ...");
     DEBUG_PRINT("sendMQTT: mqtt_enabled=");DEBUG_PRINTLN(mqtt_enabled);
     DEBUG_PRINT("sendMQTT: mqttClient.connected()=");DEBUG_PRINTLN(mqttClient.connected());
  }
  // Empty the array that strcat can work again
  mqtt_moisture_topic[0] = '\0';                  
  mqtt_battery_topic[0] = '\0';
  mqtt_version_topic[0] = '\0';
  mqtt_wakeup_topic[0] = '\0';
  mqtt_iamawake_topic[0] = '\0';
#ifdef DS18B20_SENSOR
  mqtt_temperature_topic[0] = '\0';
#endif  
}

// **************************************************************************
//  ISR for frequency counting
// **************************************************************************
ICACHE_RAM_ATTR void handleInterrupt() {
  interruptCounter++;
}

// **************************************************************************
//  Set the measure flag, called by ticker
// **************************************************************************
void set_measure_flag() {
  func_measure_call = true;
}

// **************************************************************************
//  Start frequency
// **************************************************************************
void measureFreq() {
  DEBUG_PRINTLN("Starting frequency measurement ...");
  interruptCounter = 0;                                // Initialize with 0 to avoid partly wrong measue values
  int startInt = 0;

  if (sens_enable_mode == true) {
    digitalWrite(SENSOR_ENABLE, HIGH);                 // Enable Sensor Power (High active mode)
  }
  else {
    digitalWrite(SENSOR_ENABLE, LOW);                  // Enable Sensor Power (Low active mode)
  }
    
  delay(100);                                          // Wait for sensor stabilization
  
  // Giesomat, read frequency
  startInt = millis();

  Serial.end();                                   // Serielle Uebertragung während ISR aus.
 
  // Declaration of ISR
  attachInterrupt(digitalPinToInterrupt(SENSOR_IN), handleInterrupt, RISING);
  while ((millis()-startInt) <= 40);
  detachInterrupt(digitalPinToInterrupt(SENSOR_IN));

  Serial.begin(serialspeed);                           // // Serielle Uebertragung wieder an

  freqValue = interruptCounter;
  interruptCounter = 0;
  freqValue = freqValue/40.0;
  DEBUG_PRINTLN("Frequency Sensor " + String(1) + ": " + String(freqValue) + " kHz Uptime: " + millis()+ " ms (" + String(hour()) + ":" + String(minute()) + ":" + String(second()) + ")");

#ifdef DS18B20_SENSOR
  // DS18B20 temperature measurement
  // Sensor Power on by Giesomat-Power
  DEBUG_PRINTLN("Starting measurement of temperature ...");
  DS18B20.requestTemperatures();
  temperature = DS18B20.getTempCByIndex(0);
  DEBUG_PRINTLN(String(temperature) + " °C");
  sendMQTT(MQTT_DS18B20);
#endif
  
#ifdef DEBUG  
     // delay(10000);                                      // 10s warten, um am Oszi Frequenz ablesen zu können
#endif      

  if (sens_enable_mode == true) {
    digitalWrite(SENSOR_ENABLE, LOW);                  // Disable Sensor Power
  }
  else {
    digitalWrite(SENSOR_ENABLE, HIGH);                 // Disable Sensor Power
  }
  sendMQTT(MQTT_MOISTURE);
}

// **************************************************************************
//  Self-OTA-Update
// **************************************************************************
void strremove(char* source,char ch) { 
        char* target=source;
        for (;(*target=*source)!=0;source++)
                if (*target!=ch) target++;
}

void checkForUpdates() {
  uint8_t mac[6];
  char macAddr[14];
  char chVERSION[6];
  char fwUrlBase[50];
  int FW_VERSION;  

  //Remove dots from const string VERSION
  strcpy(chVERSION, VERSION.c_str()); 
  strremove(chVERSION,'.');
  FW_VERSION =atoi(chVERSION);
    
  WiFi.macAddress( mac );
  sprintf(macAddr, "%02X%02X%02X%02X%02X%02X", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]); /// capital letters at MAC address
  strcpy(fwUrlBase, "http://");
  strcat(fwUrlBase, otau_server);
  strcat(fwUrlBase, otau_path);
  String fwURL = String( fwUrlBase );

  fwURL.concat( macAddr );
  String fwVersionURL = fwURL;
  fwVersionURL.concat( ".version" );
  
  DEBUG_PRINTLN( "Checking for firmware updates." );
  DEBUG_PRINT( "MAC address: " );
  DEBUG_PRINTLN( macAddr );
  DEBUG_PRINT( "Firmware version URL: " );
  DEBUG_PRINTLN( fwVersionURL );

  HTTPClient httpClient;
  httpClient.begin(espClient, fwVersionURL );
  int httpCode = httpClient.GET();
  if( httpCode == 200 ) {
    String newFWVersion = httpClient.getString();

    DEBUG_PRINT( "Current firmware version: " );
    DEBUG_PRINTLN( FW_VERSION );
    DEBUG_PRINT( "Available firmware version: " );
    DEBUG_PRINTLN( newFWVersion );

    int newVersion = newFWVersion.toInt();

    if( newVersion > FW_VERSION ) {
      DEBUG_PRINTLN( "Preparing to update" );

      String fwImageURL = fwURL;
      fwImageURL.concat( ".bin" );
      t_httpUpdate_return ret = ESPhttpUpdate.update(espClient, fwImageURL );

#ifdef DEBUG
      switch(ret) {
        case HTTP_UPDATE_FAILED:
          Serial.printf("HTTP_UPDATE_FAILD Error (%d): %s", ESPhttpUpdate.getLastError(), ESPhttpUpdate.getLastErrorString().c_str());
          DEBUG_PRINTLN( "Update canceled!" );
          break;

        case HTTP_UPDATE_NO_UPDATES:
          Serial.println("HTTP_UPDATE_NO_UPDATES");
          DEBUG_PRINTLN( "Update canceled!" );
          break;
      }
#endif      
    }
    else {
      DEBUG_PRINTLN( "Already on latest version" );
    }
  }
  else {
    DEBUG_PRINT( "Firmware version check failed, got HTTP response code " );
    DEBUG_PRINTLN( httpCode );
  }
  httpClient.end();
}


// **************************************************************************
//  Setup: battery measuring
// **************************************************************************
// genaue Spannung der Stromquelle, PIN A0 muss mit 5 V über ca. 180kOhm Verbunden werden! 
// Summe R=5kOhm - Spannungsteiler auf Board 100k/220k +  extern 180k
void measureBatt() { 
  
  // Read battery voltage (5000 = 5000mV)
  DEBUG_PRINTLN("Starting Battery measurement ...");

  DEBUG_PRINT("Battery voltage correction: ");
  DEBUG_PRINTLN(adc_shift);
  
  DEBUG_PRINT("Battery voltage RAW: ");
  DEBUG_PRINTLN(analogRead(A0));

  analogBattValue = String((float)(map(analogRead(A0),0,1023,0,5000))/1000+adc_shift, 3);

  DEBUG_PRINT("Battery voltage: ");
  DEBUG_PRINTLN(analogBattValue);
  sendMQTT(MQTT_BATTERY);
}

// **************************************************************************
//  Setup: Enable the Web Services, only in Power Supply Mode or when wakeup over MQTT
// **************************************************************************
void enableWebService() {    
  if (powermode || wakeup) { 
      DEBUG_PRINTLN("Enable WebService ...");
      // Enable the Free Memory Page
      httpServer.on("/freemem", []() {
      DEBUG_PRINTLN("WEB: Connection received: /freemem : ");
      DEBUG_PRINT(ESP.getFreeSketchSpace());
      httpServer.sendHeader("Connection", "close");
      httpServer.send(200, "text/plain", String(ESP.getFreeSketchSpace()).c_str());
    });

    httpServer.on("/reboot", Handle_Reboot);              // Reboot the ESP Controller
    httpServer.on("/gosleep", Handle_GoSleep);            // Reset Wakeup Topic to GoSleep after config ESP etc.
    httpServer.on("/clearconfig", Handle_ClearConfig);    // Delete the Json config file on Filesystem
    httpServer.on("/config", Handle_config);              // Show the configuration page
    httpServer.on("/getvalue", Handle_getvalue);          // Return the current frequency value as plain text without html header
    httpServer.on("/", Handle_welcome);                   // Show the main page
    httpUpdater.setup(&httpServer);                       // Enable the OTA
    httpServer.begin();                                   // Enable the WebServer
  }
}

// *************************************************************************
// Read/Write from/to RTCMemory
// *************************************************************************
void readFromRTCMemory() {
  system_rtc_mem_read(RTCMEMORYSTART, &rtcMem, sizeof(rtcMem));

  DEBUG_PRINT("Read deep_sleep_count = ");
  DEBUG_PRINTLN(rtcMem.count);
  yield();
}

void writeToRTCMemory() {
  system_rtc_mem_write(RTCMEMORYSTART, &rtcMem, sizeof(rtcMem));

  DEBUG_PRINT("Write deep_sleep_count = ");
  DEBUG_PRINTLN(rtcMem.count);
  yield();
}
// **************************************************************************
//  Setup
// **************************************************************************
void setup(void){

#ifdef DEBUG
    Serial.begin(serialspeed);
    Serial.println();
    Serial.println("Booting PSMSense Slave V " + VERSION + " ...");
#endif
  
  pinMode(CLEAR_BTN, INPUT_PULLUP);
  pinMode(SENSOR_IN, INPUT_PULLUP);
  pinMode(SENSOR_ENABLE, OUTPUT);
  pinMode(POWER_MODE, INPUT_PULLUP);

  DEBUG_PRINTLN("Reset Reason: " + String(ESP.getResetReason()));

#ifdef DS18B20_SENSOR
  // DS18B20 init
  DS18B20.begin();
#endif
  
  if (digitalRead(POWER_MODE) == HIGH) {
    powermode = true;
    DEBUG_PRINTLN("SETUP: Init wakeup counter on RTCMemory");
    readFromRTCMemory();
    if (rtcMem.count > 0) rtcMem.count = 0;               // Reset WakeUp Counter
    writeToRTCMemory();
  } else {
    powermode = false;
    if ((ESP.getResetReason().equals("Power On"))||(ESP.getResetReason().equals("External System"))) {        // Power On Reset or Reset button pressed?
      DEBUG_PRINTLN("SETUP: Reset wakeup counter after battery change or reset button press");
      rtcMem.count = 0;
    } else {
      DEBUG_PRINTLN("SETUP: Reading wakeup counter from RTCMemory");
      readFromRTCMemory();
      rtcMem.count--;
    }
    writeToRTCMemory();
  }

  if (LittleFS.begin()) {                                   // Mounting File System
    DEBUG_PRINTLN("mounted file system");
    DEBUG_PRINTLN("Load config....");
    if (!loadConfig()) {
      DEBUG_PRINTLN("Failed to load config");
    } else {
      DEBUG_PRINTLN("Config loaded");
    }
  }
  else {
    DEBUG_PRINTLN("Failed to mount FS");
  }

  if (!powermode) {
    DEBUG_PRINT("Summary sleeptime: ");
    DEBUG_PRINTLN(bat_sampling_interval);
    if (rtcMem.count == 0) {
      DEBUG_PRINT("SETUP1: Going into deep sleep for "); 
      DEBUG_PRINT(((bat_sampling_interval % 60)+1));
      DEBUG_PRINTLN(" minutes.(rtcMem.count = 0; WAKE_RFCAL)");
      ESP.deepSleep(60000000 * ((bat_sampling_interval % 60)+1), WAKE_RFCAL);      // Sleep for (bat_sampling_interval - (count*60))min and Switch on WiFi after WakeUp
      delay(100);
    } else if (rtcMem.count > 0) {
      DEBUG_PRINTLN("SETUP1: Going into deep sleep for 60 minutes.(rtcMem.count > 0; WAKE_RF_DISABLED)");
      ESP.deepSleep(60000000 * 60, WAKE_RF_DISABLED);           // Sleep for 60 min and switch off WiFi after WakeUp
      delay(100);
    }                                                           // if (rtcMem.count < 0) wakeup and work
    DEBUG_PRINTLN("Waked up and working...");
  }
  
  WiFiManager wifiManager;
  if (digitalRead(CLEAR_BTN) == LOW) wifiManager.resetSettings();   // Clear WIFI data if CLEAR Button is pressed during boot

  wifiManager.setAPCallback(configModeCallback);          // set callback that gets called when connecting to previous WiFi fails, and enters Access Point mode
  wifiManager.setSaveConfigCallback(saveConfigCallback);  // set config save notify callback
  
  if (sens_enable_mode == true) {
    digitalWrite(SENSOR_ENABLE, LOW);                // Disable Sensor Power (High active mode)
  }
  else {
    digitalWrite(SENSOR_ENABLE, HIGH);               // Disable Sensor Power (Low active mode)
  }

  // Configure some additional 
  WiFiManagerParameter custom_hostname("hostname", "Optional Hostname", host_name, 20);
  wifiManager.addParameter(&custom_hostname);
  WiFiManagerParameter custom_mqttserver("mqttserver", "Optional MQTT Server", mqtt_server, 30);
  wifiManager.addParameter(&custom_mqttserver);
  WiFiManagerParameter custom_mqtttopic("mqtttopic", "Optional MQTT Topic", mqtt_topic, 30);
  wifiManager.addParameter(&custom_mqtttopic);
  WiFiManagerParameter custom_ntpserver("ntpserver", "Optional NTP Server", ntpserver, 30);
  wifiManager.addParameter(&custom_ntpserver);
  WiFiManagerParameter custom_otauserver("otuaserver", "Optional OTA-Update Server", otau_server, 30);
  wifiManager.addParameter(&custom_otauserver);

  String autoconf_ssid = "PSMSense_Config_"+String(ESP.getChipId());
  wifiManager.setConnectTimeout(60);                                  // Workaround Test for reconnect issue
  wifiManager.setConfigPortalTimeout(120);                            // Timeout for SoftAP, try connect again to stored wlan
  wifiManager.autoConnect(autoconf_ssid.c_str());                     // Use PSMSense_Config_+Chip ID as AP-name with 192.168.4.1 as IP
  
  // Read and save the new values from AP config page
  strncpy(host_name, custom_hostname.getValue(), 20);
  strncpy(mqtt_server, custom_mqttserver.getValue(), 30);
  strncpy(mqtt_topic, custom_mqtttopic.getValue(), 30);
  strncpy(ntpserver, custom_ntpserver.getValue(), 30);
  strncpy(otau_server, custom_otauserver.getValue(), 30);

  DEBUG_PRINTLN("WiFi connected! IP: " + ipToString(WiFi.localIP()) + " Hostname: " + String(host_name));
  DEBUG_PRINTLN("NTP-Server: " + String(ntpserver) + " MQTT Server: " + String(mqtt_server) + " MQTT Topic: " + String(mqtt_topic) + "/?/sensor/<sensor_number/>" );
  DEBUG_PRINTLN("OTAU-Server: " + String(otau_server));

  // save the custom parameters to FS
  if (shouldSaveConfig) {
    DEBUG_PRINTLN(" config...");
    DynamicJsonBuffer jsonBuffer;
    JsonObject& json = jsonBuffer.createObject();
    json["hostname"] = host_name;
    json["mqttserver"] = mqtt_server;
    json["mqtttopic"] = mqtt_topic;
    json["ntpserver"] = ntpserver;
    json["otauserver"] = otau_server;

    File configFile = LittleFS.open("/config.json", "w");
    if (!configFile) {
      DEBUG_PRINTLN("SPI: failed to open config file for writing config");
    }
    #ifdef DEBUG
      json.printTo(Serial);
    #endif
    DEBUG_PRINTLN("");
    json.printTo(configFile);
    configFile.close();
  }

  if (host_name[0] == 0 ) strncpy(host_name, "PSMSense", 20);         //set default hostname when not set!
  if (mqtt_topic[0] == 0 ) strncpy(mqtt_topic, defaultmqtttopic, 30); //set defaultmqtttopic when not set!
  if (ntpserver[0] == 0 ) strncpy(ntpserver, poolServerName, 30);     //set default ntp server when not set!
  
  //MDNS.begin(host);

  enableWebService();

// **************************************************************************
//  NTP
// **************************************************************************

  if (ntp_enabled) {
      DEBUG_PRINTLN("NTP: Starting UDP");
      Udp.begin(NtpLocalPort);
      DEBUG_PRINT("NTP: Local port: ");
      DEBUG_PRINTLN(Udp.localPort());
      DEBUG_PRINTLN("NTP: waiting for sync");
      setSyncProvider(getNtpTime);                        // set the external time provider
      setSyncInterval(3600);                             // set the number of seconds between re-sync
      //String boottimetemp = printDigits2(hour()) + ":" + printDigits2(minute()) + " " + printDigits2(day()) + "." + printDigits2(month()) + "." + String(year());
      //strncpy(boottime, boottimetemp.c_str(), 20);     // If we got time set boottime
  } else {
      setTime(1609455600);  // 1.1.2021 00:00 Initialize time
  }

// **************************************************************************
//  Setup: MQTT
// **************************************************************************
  if ((mqtt_enabled) && (strlen(mqtt_server) > 2)){            // Broker defined?
     mqttClient.setServer(mqtt_server, 1883);                  // Set the MQTT Broker
     mqttClient.setCallback(callbackMQTT);
//     mqttClient.loop();
//     delay(10);
//     sendMQTT(MQTT_WAKEUP);
  }

// **************************************************************************
//  Send Battery-Status and Sensor-Fruency; Check for Update
// **************************************************************************

  // Check state bevor sendMQTT()
  measureBatt();                                         // check battery voltage -> sendMQTT
  measureFreq();                                         // check frequency value and temperature -> sendMQTT
  millisecs = millis();                                  // start millisecs

  DEBUG_PRINT("SETUP: Check for Update (otau_enabled): ");
  DEBUG_PRINTLN(otau_enabled);

  if (otau_enabled) { 
    DEBUG_PRINTLN("SETUP: Check for Update ...");
    checkForUpdates();                                    // Check for Updates on HTTP-Server -> sendMQTT
  } else {
    DEBUG_PRINTLN("SETUP: Update disabled ...");
  }

// **************************************************************************
//  Sleep-Mode
// **************************************************************************
      
  // Only in Power Supply Mode
  if (powermode) {
    DEBUG_PRINTLN("SETUP: Startup completed in Power Supply mode ...");
    measureFreq();                                                      // Start first measure immediately
    startMeasure.attach(power_sampling_interval*60, set_measure_flag);  // Start measure periodically each configured interval time (set the flag)
  } 
  else if ((millis() - millisecs) > mqtt_receive_wait ){
    DEBUG_PRINTLN("SETUP2: Startup completed in Battery mode ...");
    measureFreq();                                                     // Start measure and send via MQTT if enabled
    sendMQTT(MQTT_VERSION);
    DEBUG_PRINT("SETUP2: Going into deep sleep mode for "); 
    DEBUG_PRINT((bat_sampling_interval > 60)? 60:bat_sampling_interval); 
    if (bat_sampling_interval > 60) {
       DEBUG_PRINT(" minutes.(rtcMem.count = "); 
       DEBUG_PRINT(rtcMem.count); 
       DEBUG_PRINTLN(";WAKE_RF_DISABLED)");
       rtcMem.count = bat_sampling_interval/60;                         // Load counter for full houres
       DEBUG_PRINTLN("SETUP2: Writing wakup counter to RTCMemory");
       writeToRTCMemory();                                              // New value after rtcMem.count <= 0
       ESP.deepSleep(60000000 * 60, WAKE_RF_DISABLED);                  // 60s * min one houre without WiFi
    } else {
       DEBUG_PRINTLN(" minutes.(rtcMem.count = 0; WAKE_RFCAL)");
       ESP.deepSleep(60000000* bat_sampling_interval, WAKE_RFCAL);      // 60s * Value in Minutes < 60 from config with WiFi
    }
    delay(100);                                                         // Only to ensure the Deepsleep is working
  } else {
    DEBUG_PRINTLN("SETUP2: Going into loop mode ...");
  }
}

// **************************************************************************
//  Main Loop
// **************************************************************************
void loop(void){
  // Only in Power Supply Mode
  //if (powermode) { 
    httpServer.handleClient();

    if ((mqtt_enabled) && (strlen(mqtt_server) > 2)){
      if (!mqttClient.connected()) {
        reconnectMQTT();
      }
//      DEBUG_PRINTLN("LOOP: Check for MQTT subscribes... "); 
      mqttClient.loop();
      delay(10);                        // Wait for receive messages
    }

    if (func_measure_call) {            // If the call function flag is set by ticker, execute measureFreq()
      measureBatt();                    // check battery voltage
      measureFreq();                    // Giesomat and temperature masurement (if enabled) 
      sendMQTT(MQTT_VERSION);
      func_measure_call = false;
    }
    
//  }
  if (!powermode){ // Battery Mode
    if ((!wakeup) &&  ((millis() - millisecs) > mqtt_receive_wait )) {     // If wakeup reset by mqtt and subscribed topic received go sleep
       sendMQTT(MQTT_VERSION);
       DEBUG_PRINT("LOOP: Going into deep sleep mode for "); 
       DEBUG_PRINT((bat_sampling_interval > 60)? 60:bat_sampling_interval); 
       if (bat_sampling_interval > 60) {
          DEBUG_PRINT(" minutes.(rtcMem.count = "); 
          DEBUG_PRINT(rtcMem.count); 
          DEBUG_PRINTLN(";WAKE_RF_DISABLED)");
          rtcMem.count = bat_sampling_interval/60;                         // Load counter for full houres
          DEBUG_PRINTLN("LOOP: Writing wakup counter to RTCMemory");
          writeToRTCMemory();                                              // New value after rtcMem.count <= 0
          ESP.deepSleep(60000000 * 60, WAKE_RF_DISABLED);                  // 60s * min one houre without WiFi
       } else {
          DEBUG_PRINTLN(" minutes.(rtcMem.count = 0; WAKE_RFCAL)");
          rtcMem.count = 0;                                                // Load counter for full houres
          DEBUG_PRINTLN("LOOP: Writing wakup counter=0 to RTCMemory");
          writeToRTCMemory();                                              // New value after rtcMem.count <= 0
          ESP.deepSleep(60000000 * bat_sampling_interval, WAKE_RFCAL);      // 60s * Value in Minutes < 60 from config with WiFi
       }
       delay(100);                                                         // Only to ensure the Deepsleep is working
    }  
  }
}
// **************************************************************************
//  EOF
// **************************************************************************
