// ESP Gies-O-Mat MQTT Soil Moisture Controller
// Plant Soil Moisture Sense (PSMSense) - Receiver
// Master
// 
// © Copyright 2020
// Daniel Willhelm 
// Ralph Lautenschläger
//
//
// Arduino Config
// LOLIN (WEMOS) D1 mini
// 4M (1M SPIFFS);v2 Prebuild (MSS=536); 160MHz; 
// Arduino IDE 1.8.16
// ESP8266 3.0.2
// WIFIManager 2.0.4-beta
// ArduinoJson V5
// PubSubClient 2.8.0
// 
// https://github.com/esp8266/Arduino/blob/master/variants/d1_mini/pins_arduino.h
// static const uint8_t D0   = 16;
// static const uint8_t D1   = 5;
// static const uint8_t D2   = 4;
// static const uint8_t D3   = 0;
// static const uint8_t D4   = 2;
// static const uint8_t D5   = 14;
// static const uint8_t D6   = 12;
// static const uint8_t D7   = 13;
// static const uint8_t D8   = 15;
// static const uint8_t RX   = 3;
// static const uint8_t TX   = 1;
// 
//Frontend
//Frontend open-source toolkit 3.3.7
//https://getbootstrap.com/
//
// ToDo
// - formating output tag in sensor config page
//
// Bugs
// - 
//
// 0.1.0 First stable working version (@160MHz)
// 0.2.0 Added bistabil relays functions
// 0.2.1 Added multisense relays functions
// 2.1.1 Added buildnumber autoincrement
// 2.2.2 Added self OTA update
// 2.3.1 Added sensor config page
// 2.4.199 Added ADC-Shift correction to configuration-page
// 2.5.217 Added OTA-Update Configuration to configuration-page
// 2.6.244 Added MQTT-Topic-receive wait counter befor goto deepsleep
// 2.7.290 Added wakeup via MQTT
// 2.8.298 Added I am awake topic (Set if received wakeup command end reset if press gosleep button)
// 2.8.300 Change deep_sleep() type in ESP.h,ESP.cpp from uint64_t in uint32_t to use max deep_sleep time of 71 minutes
//         (http://www.edaboard.de/esp-deep-sleep-bug-t24751.html) Error 404
//         (https://www.edaboard.de/threads/esp-deep_sleep-bug.11152290/#post-37612763) Bssucht 12.02.2021
// 2.9.301 Added topic to enable/disable calculated, humidity dependent control
// 3.0.312 Added second sensor output
// 3.0.365 Added SysConfig Page
// 3.0.402 Change ESPcore to 2.7.4
// 3.0.403 Added treshold MQTT publishing
// 3.0.406 Added topic ".../last_transmission/master"
// 3.1.421 Added MQTT topic "/humid/master/averageTresholdValueGroup/xx"
// 3.1.434 Added MQTT publishing enable/disable for test (#define MQTT_PUBLISH)
// 3.1.457 Added MQTT receive latency multiplied by number of sensors and any optimized debug outputs
// 3.1.460 Added MQTT error message if relays don't switch -> battery low
// 3.1.473 Added time range in configuration in which relays are not allowed to switch
// (When irrigation cycle has started, it should not be interrupted)
// 3.2.517 Added customizable threshold for sensor groups
// 3.2.534 Added subfunction for sendMQTT() call
// 3.2.541 Added ntp_sync flag 
// 3.2.563 Added Condition that averageTresholdValueGroup is only published via MQTT when all sensor values are received. 
//               This prevents intermediate values in the home automation database.
// 3.2.571 Changed: averageTresholdValueGroup1/2 stored in rtcMem
// 3.3.575 Change ESPcore to 3.0.2
// 3.3.584 Change FS.h to LitleFS.h -> SPIFFS to LittleFS
// 3.3.595 Split htmlSensorconf in tree parts becouse TCP buffer overflow (https://github.com/esp8266/Arduino/issues/3205)
// 3.4.600 Use the last transmission of sensors to calculate without sensors that have not transmitted for more than 6 hours.
// 3.4.607 Optimizion if mqtt ist disabled
// *************************************************************************
//  Includes
// **************************************************************************
#include "version.h"
#include <ESP8266WiFi.h>                // 
#include <DNSServer.h>                  // Local DNS Server used for redirecting all requests to the configuration portal
#include <ESP8266WebServer.h>           // Local WebServer used to serve the configuration portal
#include <ESP8266HTTPClient.h>          // HTTP Client for Self-OTA-Update
#include <ESP8266httpUpdate.h>          // Used for Self-OTA-Update
#include <WiFiManager.h>                // https://github.com/tzapu/WiFiManager WiFi Configuration Magic
#include <ESP8266HTTPUpdateServer.h>
#include <Ticker.h>                     // Periodically call of a function
#include <ArduinoJson.h>                // Config file handling
#include <TimeLib.h>                    // https://github.com/PaulStoffregen/Time
#include <Timezone.h>                   // https://github.com/JChristensen/Timezone
#include "LittleFS.h"

extern "C" {
#include "user_interface.h"             // Used for RTC Memory access
}

#define MQTT_VERSION MQTT_VERSION_3_1   // That's needed to change in PubSubClient.h !!!
#include <PubSubClient.h>               // MQTT 2.8.0

ESP8266WebServer httpServer(80);
ESP8266HTTPUpdateServer httpUpdater;

Ticker startMeasure;

WiFiClient espClient;                   // Only needed for MQTT PubSubClient!
PubSubClient client(espClient);         // Enable MQTT

WiFiUDP Udp;                            // Needed fpr NTP
unsigned int NtpLocalPort = 123;        // local port to listen for UDP packets (NTP)

// **************************************************************************
//  Defines 
// **************************************************************************
#define DEBUG                         // Define for Debug output on serial interface   
#define MQTT_PUBLISH                  // Define for disabling or enabling MQTT publishing (comment out for Test)
#define WAKEUP                  D0    // Wakeup-Output (GPIO16)
#define SET_RELAYS1             D4    // Output 0 Ohm for Sensor 1 (GPIO2)
#define RESET_RELAYS1           D1    // Output 10 kOhm for Sensor 1 (GPIO5)
#define SET_RELAYS2             D3    // Output 0 Ohm for Sensor 2 (GPIO0)
#define RESET_RELAYS2           D2    // Output 10 kOhm for Sensor 2 (GPIO4)
#define CLEAR_BTN               D2    // LOW during PowerUp will clear the json config file (GPIO4)
#define BATT_IN_ANALOG          A0    // Analog Battery messure (ADC)
#define POWER_MODE              D6    // High for Power Supply, Low for Battery Mode (GPIO12)
#define STATUS_1_IN_0           D5    // Low for Status 0 Ohm for Sensor 1 -> wet (GPIO14)    -> don't watering
                                      // High for Status 10 kOhm for Sensor 1 -> dry (GPIO14) -> watering
#define STATUS_2_IN_0           D7    // Low for Status 0 Ohm for Sensor 2 -> wet (GPIO13)    -> don't watering                                                                                                                                                                                                                              // High for Status 0 Ohm for Sensor 2 -> dry (GPO13)
                                      // High for Status 10 kOhm for Sensor 2 -> dry (GPIO13) -> watering

// **************************************************************************
// Use RTC for extend deep_sleep_Counter
// Avarage Value for pubish over MQTT and use to display on Diagramm 
// **************************************************************************
#define RTCMEMORYSTART          65    // First RTC Memoryposition
typedef struct {
  float averageTresholdValueGroup1 = 0.0;
  float averageTresholdValueGroup2 = 0.0;
} rtcStore;
rtcStore rtcMem;
//*****************************************************************************
                                        
const String FIRMWARE_NAME = "PSM-Sense-Master";
const String VERSION       = SKETCH_VERSION;
//const char* fwUrlBase = "http://192.168.XXX.XXX/PSMSenseReceiver/";

// *************************************************************************
// SendMQTT Defines
// *************************************************************************
#define MQTT_LASTTRANSMISSION               // sendMQTT()
#define MQTT_ALL                      0xFF  // not in use yet
#define MQTT_TRESHOLD                 0x01  // Handle_sensor_config()
#define MQTT_STATE                    0x02  // getRelayState()
#define MQTT_AVERAGETRESHOLD          0x04  // outputMoisture()
#define MQTT_VERSION                  0x08  // checkForUpdates()
#define MQTT_BATTERY                  0x10  // measureBatt() 
#define MQTT_WAKEUP                   0x20  // Handle_GoSleep()
#define MQTT_IMAWAKE                  0x40  // callbackMQTT()
#define MQTT_ERROR                    0x80  // outputMoisture()

// **************************************************************************
//  Debug
// **************************************************************************
#ifdef DEBUG
#define DEBUG_PRINT(x)  Serial.print (x)
#define DEBUG_PRINTLN(x)  Serial.println (x)
#else
#define DEBUG_PRINT(x)
#define DEBUG_PRINTLN(x)
#endif

// **************************************************************************
//  Variables
// **************************************************************************
// config.json
char host_name[20] = "";
char ntpserver[30] = "";
char otau_server[30] = "";                          // Server for OTA-Updates
char otau_path[30] = "/PSMSenseReceiver/";          // Path to Updatefile
bool otau_enabled = false;
char mqtt_server[30] = "";
char mqtt_topic[30] = "";
bool mqtt_enabled = false;
bool mqtt_topic_received = false;                    // wait for receive topic befor goto deepsleep
int sensors_count = 1;                               // How many sensors connected
int mqtt_receive_wait = 1500;                        // time to waiting for mqtt-subscribes (1,5s/Sensor)
int millisecs = 0;                                   // milisecs by programmstart                                                 
bool gosleep = false;                                // True if call /gosleep or press Button "GoSleeep" on WebPage
char defaultmqtttopic[30] = "PSMSense/moisture/sensor/";
int power_sampling_interval = 10;                    // Measure Frequency in Power Mode in Minutes
int bat_sampling_interval = 10;                      // Measure Frequency in Battery Mode in Minutes
float adc_shift=0.00;                                // Battery voltage correction
char blockingtime_from[6] = "00:00";                 // Start time to suppress switching of the relays
char blockingtime_to[6] = "23:59";                   // End time to suppress switching of the relays

// system_config.json
// float averageTresholdValueGroup1 = 0.0;         // Avarage Value for pubish over MQTT and use to display on Diagramm 
// float averageTresholdValueGroup2 = 0.0;         // Avarage Value for pubish over MQTT and use to display on Diagramm 

//Sensorconfig
bool sensorAvailable[10];                            // Consider sensor when calculating the average
int sensorWeighting[10];                             // Weighting of the sensor when calculating the average value
int sensorTreshold[10];                              // defines the threshold value dry/wet
int sensorGroup[10];                                 // Save the sensor assignment to sensor input 1 or 2
char sensorName[10][25];                             // Name of Seonsor
float tresholdGroup1 = 0.0;                          // Treshold for watering group 1
float tresholdGroup2 = 0.0;                          // Treshold for watering group 2

//NTP
bool ntp_enabled = false;                            // Set to false to disable querying for the time
bool ntp_sync = false;                               // if getNTPTime() received time
char poolServerName[30] = "europe.pool.ntp.org";     // default NTP Server when not configured in config.json
char boottime[20] = "";                              // PSMSene Boot Time
const int timeZone = 1;                              // Central European Time
const int NTP_PACKET_SIZE = 48;                      // NTP time is in the first 48 bytes of message
byte packetBuffer[NTP_PACKET_SIZE];                  // Buffer to hold incoming & outgoing packets

//Timezone
//Central European Time (Frankfurt, Paris)
TimeChangeRule CEST = { "CEST", Last, Sun, Mar, 2, 120 };     //Central European Summer Time
TimeChangeRule CET = { "CET ", Last, Sun, Oct, 3, 60 };       //Central European Standard Time
Timezone CE(CEST, CET);
TimeChangeRule *tcr;        //pointer to the time change rule, use to get the TZ abbrev
time_t utc, local;


// Webserver
String javaScript;
String htmlHeader;
String htmlFooter;

// Other
float freqValue[10] = {0};                      // Sensor Frequency (max. 10 Slaves)
char arrivingTime[10][25] = {0};                // Last arriving (max. 10 Slaves)
float analogValue[11] = {0};                    // Battery Voltage (max. 10 Slaves + 1 Master)
bool receivedFreqValue[10] = {false};           // Receive Flag Sensor Frequency (max. 10 Slaves)
//bool bSensorAvailable[10] = {false};            // Store availability of max 10 sensors
bool shouldSaveConfig = false;                  // Flag for saving data
bool powermode = true;                          // Power Mode True for Power Supply, False for Battery Mode
bool func_measure_call = false;                 // Flag for jump into the measure function
bool disableTresholdCalculation = false;        // Disable switching in time range of sprinkeling automatic 
bool relays1_state = false;                     // true=wet, false=dry for relays 1
bool relays2_state = false;                     // true=wet, false=dry for relays 2
bool dont_switch_error1 = false;                // set to true if relais 1 has not switched. battery voltage to low
bool dont_switch_error2 = false;                // set to true if relais 2 has not switched. battery voltage to low
bool enableHumidCalculationSensor1 = true;      // If tru, dry/wet state based on calculation. 
                                                // If false, state goes to wet to disabled MultiControl programm, e.g. while its raining.
bool enableHumidCalculationSensor2 = true;      // If tru, dry/wet state based on calculation. 
                                                // If false, state goes to wet to disabled MultiControl programm, e.g. while its raining.                                                
bool wakeup = false;                            // Set/Reset wakeup flag via mqtt for configuration
bool iamawake = false;                          // Set/Reset iamawake flag if ESP received wakeup command
int serialspeed = 74880;                        // Baud rate for Serial.print (boot messages are send via 74880 baud)

// **************************************************************************
// JSON Configuration Management
// **************************************************************************
bool loadConfig() {
  if (LittleFS.exists("/config.json")) {
      File configFile = LittleFS.open("/config.json", "r");
      if (configFile) {
        DEBUG_PRINT("SPI (loadConfig): opened config file - size [Byte]: ");
        size_t size = configFile.size();
        DEBUG_PRINTLN(size);
        if (size > 1024) {
          DEBUG_PRINTLN("SPI (loadConfig): Config file size is too large");
          return false;
        }
        std::unique_ptr<char[]> buf(new char[size]);            // Allocate a buffer to store contents of the file.
        configFile.readBytes(buf.get(), size);                  // Input Buffer (needed by ArduinoJson)
        DynamicJsonBuffer jsonBuffer;
        JsonObject& json = jsonBuffer.parseObject(buf.get());
        #ifdef DEBUG
          json.printTo(Serial);
        #endif
        if (json.success()) {
          DEBUG_PRINTLN("\nJSO (loadConfig): parsed config.json:");
          if (json.containsKey("hostname")) strncpy(host_name, json["hostname"], 20);
          if (json.containsKey("mqttserver")) strncpy(mqtt_server, json["mqttserver"], 30);
          if (json.containsKey("mqtttopic")) strncpy(mqtt_topic, json["mqtttopic"], 30);
          if (json.containsKey("mqttenabled")) mqtt_enabled = json["mqttenabled"];
          if (json.containsKey("ntpserver")) strncpy(ntpserver, json["ntpserver"], 30);
          if (json.containsKey("ntpenabled")) ntp_enabled = json["ntpenabled"];
          if (json.containsKey("otauserver")) strncpy(otau_server, json["otauserver"], 30);
          if (json.containsKey("otauenabled")) otau_enabled = json["otauenabled"];
          if (json.containsKey("blockingtimefrom")) strncpy(blockingtime_from, json["blockingtimefrom"], 6);
          if (json.containsKey("blockingtimeto")) strncpy(blockingtime_to, json["blockingtimeto"], 6);
          if (json.containsKey("sampleintpower")) power_sampling_interval = json["sampleintpower"];
          if (json.containsKey("sampleintbat")) bat_sampling_interval = json["sampleintbat"];
          if (json.containsKey("sensorscount")) sensors_count = json["sensorscount"];
          if (json.containsKey("adc_shift")) adc_shift = json["adc_shift"];

          if (mqtt_server == "") mqtt_enabled = 0;
          DEBUG_PRINT("loadConfig: Read json[otauenabled]: ");
          DEBUG_PRINTLN(otau_enabled);
        } else {
          DEBUG_PRINTLN("SPI (loadConfig): Failed to parse config file");
          return false;
        }
      } else {
        DEBUG_PRINTLN("SPI (loadConfig): Failed to open config file");
        return false;
      }
  }
  return true;
}

// **************************************************************************
// JSON Sensor Configuration Management
// **************************************************************************
bool loadSensorConfig() {
  char sensor_index[][3] = {"00","01","02","03","04","05","06","07","08","09","10"};
  char jsonSensorName[20];
  char jsonSensorEnabled[20];
  char jsonSensorGroup[20];
  char jsonAverageWeight[20];
  char jsonMoisTreshold[20];

  if (LittleFS.exists("/sensor_config.json")) {
      File configFile = LittleFS.open("/sensor_config.json", "r");
      if (configFile) {
         DEBUG_PRINT("SPI (loadSensorConfig): opened sensor_config file - size [Byte]: ");
         size_t size = configFile.size();
         DEBUG_PRINTLN(size);
         if (size > 1024) {
           DEBUG_PRINTLN("SPI (loadSensorConfig): Master-Config file size is too large");
           return false;
         }
         std::unique_ptr<char[]> buf(new char[size]);            // Allocate a buffer to store contents of the file.
         configFile.readBytes(buf.get(), size);                  // Input Buffer (needed by ArduinoJson)
         DynamicJsonBuffer jsonBuffer;
         JsonObject& json = jsonBuffer.parseObject(buf.get());
         #ifdef DEBUG
           json.printTo(Serial);
         #endif
         if (json.success()) {
            DEBUG_PRINTLN("\nJSO (loadSensorConfig):parsed json");
            for (int i = 1; i <= sensors_count ; i++) {
               DEBUG_PRINT("JSO (loadSensorConfig):Read Sensor_Index: ");
               DEBUG_PRINTLN(sensor_index[i]);

               strcpy(jsonSensorName, "sensor_name");
               strcat(jsonSensorName, sensor_index[i]);
               if (json.containsKey(jsonSensorName)) strncpy(sensorName[i], json[jsonSensorName],20);
               strcpy(jsonSensorEnabled, "sensorenabled");
               strcat(jsonSensorEnabled, sensor_index[i]);
               if (json.containsKey(jsonSensorEnabled)) sensorAvailable[i] = json[jsonSensorEnabled];
               strcpy(jsonSensorGroup, "sensorgroup");
               strcat(jsonSensorGroup, sensor_index[i]);
               if (json.containsKey(jsonSensorGroup)) sensorGroup[i] = json[jsonSensorGroup];
               strcpy(jsonAverageWeight, "average_weight");
               strcat(jsonAverageWeight, sensor_index[i]);
               if (json.containsKey(jsonAverageWeight)) sensorWeighting[i] = json[jsonAverageWeight];
               strcpy(jsonMoisTreshold, "mois_treshold");
               strcat(jsonMoisTreshold, sensor_index[i]);
               if (json.containsKey(jsonMoisTreshold)) sensorTreshold[i] = json[jsonMoisTreshold];
           }  
           if (json.containsKey("tresholdGroup1")) tresholdGroup1 = json["tresholdGroup1"];
           if (json.containsKey("tresholdGroup2")) tresholdGroup2 = json["tresholdGroup2"];
        } else {
           DEBUG_PRINTLN("SPI (loadSensorConfig): Failed to parse config file");
           return false;
        }
     } else {
         DEBUG_PRINTLN("SPI (loadSensorConfig): Failed to open config file");
         return false;
     }
  }
  return true;
}


// **************************************************************************
//  Callback notifying us of the need to save config
// **************************************************************************
void saveConfigCallback ()
{
  DEBUG_PRINTLN("Should save config");
  shouldSaveConfig = true;
}

// **************************************************************************
//  Gets called when WiFiManager enters configuration mode
// **************************************************************************
void configModeCallback (WiFiManager *myWiFiManager)
{
  DEBUG_PRINTLN("Entered config mode");
  DEBUG_PRINTLN(WiFi.softAPIP());
  DEBUG_PRINTLN(myWiFiManager->getConfigPortalSSID());    //if you used auto generated SSID, print it
}


// **************************************************************************
//  IP Address to String
// **************************************************************************
String ipToString(IPAddress ip)
{
  String s = "";
  for (int i = 0; i < 4; i++)
    s += i  ? "." + String(ip[i]) : String(ip[i]);
  return s;
}


// **************************************************************************
//  NTP Time Syncronization
// **************************************************************************
time_t getNtpTime()
{
  IPAddress ntpServerIP;                            // NTP server's ip address

  while (Udp.parsePacket() > 0) ;                   // Discard any previously received packets
  DEBUG_PRINTLN("NTP: Transmit NTP Request");
  WiFi.hostByName(ntpserver, ntpServerIP);          // Lookup IP from Hostname
  DEBUG_PRINT("NTP: ");
  DEBUG_PRINT(ntpserver);
  DEBUG_PRINT(" IP: ");
  DEBUG_PRINTLN(ntpServerIP);
  sendNTPpacket(ntpServerIP);
  uint32_t beginWait = millis();
  while (millis() - beginWait < 1500) {
    int size = Udp.parsePacket();
    if (size >= NTP_PACKET_SIZE) {
      DEBUG_PRINTLN("NTP: Receive NTP Response");
      Udp.read(packetBuffer, NTP_PACKET_SIZE);  // read packet into the buffer
      unsigned long secsSince1900;
      // convert four bytes starting at location 40 to a long integer
      secsSince1900 =  (unsigned long)packetBuffer[40] << 24;
      secsSince1900 |= (unsigned long)packetBuffer[41] << 16;
      secsSince1900 |= (unsigned long)packetBuffer[42] << 8;
      secsSince1900 |= (unsigned long)packetBuffer[43];
      secsSince1900 -= 2208988800UL;  //+ (timeZone * 60 * 60); // NTP Time - 70 Years + TimeZone Hours
      ntp_sync = true;
      return CE.toLocal(secsSince1900, &tcr);
    }
  }
  DEBUG_PRINTLN("NTP: No NTP Response :-(");
  ntp_sync = false;
  return 0;                                          // return 0 if unable to get the time
}
// **************************************************************************
// send an NTP request to the time server at the given address
// **************************************************************************
void sendNTPpacket(IPAddress &address)
{
  memset(packetBuffer, 0, NTP_PACKET_SIZE);                 // set all bytes in the buffer to 0
  packetBuffer[0] = 0b11100011;                             // LI (Clock is unsynchronized), Version (4), Mode (Client)
  packetBuffer[1] = 0;                                      // Stratum, or type of clock (Unspecified or invalid)
  packetBuffer[2] = 6;                                      // Polling Interval (log2 seconds)
  packetBuffer[3] = 0xEC;                                   // Peer Clock Precision (log2 seconds)
                                                            // 8 bytes of zero for Root Delay & Root Dispersion
  packetBuffer[12] = 49;                                    // 
  packetBuffer[13] = 0x4E;                                  //
  packetBuffer[14] = 49;                                    //
  packetBuffer[15] = 52;                                    //
  Udp.beginPacket(address, 123);                            // NTP requests are to port 123
  Udp.write(packetBuffer, NTP_PACKET_SIZE);
  Udp.endPacket();
}

// **************************************************************************
//  HTML Header Template
// **************************************************************************
void buildHeader()
{
  htmlHeader="<!DOCTYPE html PUBLIC '-//W3C//DTD XHTML 1.0 Strict//EN' 'http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd'>\n";
  htmlHeader+="<html xmlns='http://www.w3.org/1999/xhtml' xml:lang='en'>\n";
  htmlHeader+="  <head>\n";
  htmlHeader+="    <meta name='viewport' content='width=device-width, initial-scale=.75' />\n";
  htmlHeader+="    <meta http-equiv='Cache-Control' content='no-cache, no-store, must-revalidate' />\n";
  htmlHeader+="    <meta http-equiv='Pragma' content='no-cache' />\n";
  htmlHeader+="    <meta http-equiv='Expires' content='0' />\n";
  htmlHeader+="    <link rel='stylesheet' href='https://maxcdn.bootstrapcdn.com/bootstrap/3.3.7/css/bootstrap.min.css' />\n";
  htmlHeader+="    <style>@media (max-width: 991px) {.nav-pills>li {float: none; margin-left: 0; margin-top: 5px; text-align: center;}}</style>\n";
  htmlHeader+="    <title>" + FIRMWARE_NAME + " - V " + VERSION + "</title>\n";
  htmlHeader+="  </head>\n";
  htmlHeader+="  <body>\n";
  htmlHeader+="    <div class='container'>\n";
  htmlHeader+="      <h1>" + FIRMWARE_NAME + " - V " + VERSION + "</h1>\n";
  htmlHeader+="      <div class='row'>\n";
  htmlHeader+="        <div class='col-md-12'>\n";
  htmlHeader+="          <ul class='nav nav-pills'>\n";
  htmlHeader+="            <li class='active'>\n";
  htmlHeader+="              <a href='#'>Hostname <span class='badge'>" + String(host_name) + "</span></a></li>\n";
  htmlHeader+="            <li class='active'>\n";
  htmlHeader+="              <a href='http://" + ipToString(WiFi.localIP()) + ":80'>Local IP<span class='badge'>" + ipToString(WiFi.localIP()) + "</span></a></li>\n";
  htmlHeader+="            <li class='active'>\n";
  htmlHeader+="              <a href='#'>MAC <span class='badge'>" + String(WiFi.macAddress()) + "</span></a></li>\n";
  htmlHeader+="            <li class='active'>\n";
  htmlHeader+="              <a href='/config'>Configuration</a></li>\n";
  htmlHeader+="            <li class='active'>\n";
  htmlHeader+="              <a href='#'><span class='glyphicon glyphicon-signal'></span> "+ String(WiFi.RSSI()) + " dBm</a></li>\n";
  htmlHeader+="          </ul>\n";
  htmlHeader+="        </div>\n";
  htmlHeader+="      </div><hr />\n";
}


// **************************************************************************
//  HTML Footer Template
// **************************************************************************
void buildFooter()
{
  char nowTime[10];
  
  sprintf(nowTime, "%02d:%02d:%02d", hour(now()), minute(now()), second(now()));
  htmlFooter="       <br><br><hr />";
  htmlFooter+="      <div class='row'><div class='col-md-12'><em>MQTT Plant Soil Moisture Master Gies-O-Mat -- V " + VERSION + " -- Zeit: " + nowTime +"</em></div></div>\n";
  if (client.connected() &&  mqtt_enabled) {
      htmlFooter+="      <div class='row'><div class='col-md-12'><em>MQTT Broker: " + String(mqtt_server) + "</em></div></div>\n";
  } else {
      htmlFooter+="      <div class='row'><div class='col-md-12'><em>MQTT Broker: not connected or disabled</em></div></div>\n";
  }
  if (otau_enabled) {
      htmlFooter+="      <div class='row'><div class='col-md-12'><em>OTA-Update Server: http://" + String(otau_server) + String(otau_path) + "</em></div></div>\n";
  } else {
      htmlFooter+="      <div class='row'><div class='col-md-12'><em>OTA-Update: disabled </em></div></div>\n";
  }
  htmlFooter+="    </div>\n";
  htmlFooter+="  </body>\n";
  htmlFooter+="</html>\n";
}

// **************************************************************************
//  HTML Page for ESP Reboot
// **************************************************************************
void Handle_Reboot()
{
  httpServer.sendHeader("Connection", "close");
  httpServer.send(200, "text/html", F("<body>Reboot OK, please reload page.</body>"));
  // make sure that GPIO0 gets the high level via pullup at start
  pinMode(SET_RELAYS1, INPUT);
  pinMode(SET_RELAYS2, INPUT);
  delay(500);
  ESP.reset();
}

// **************************************************************************
//  HTML Page for ESP Reboot
// **************************************************************************
void Handle_GoSleep()
{
  httpServer.sendHeader("Connection", "close");
  httpServer.send(200, "text/html", F("<body>Go sleeping. Good night!</body>"));
  delay(500);
  gosleep = true;  
  sendMQTT(MQTT_WAKEUP);   // Reset wakeup topic
  // make sure that GPIO0 gets the high level via pullup at start
  pinMode(SET_RELAYS2, INPUT);
  delay(500);
  ESP.restart();
}

// **************************************************************************
//  HTML Page for ESP Clear Json config-file
// **************************************************************************
void Handle_ClearConfig()
{
  httpServer.sendHeader("Connection", "close"); 
  httpServer.send(200, "text/html", F("<body>Config File deleted, ESP reboot, please reload page.</body>"));
  LittleFS.remove("/config.json");
  LittleFS.remove("/sensor_config.json");
  // make sure that GPIO0 gets the high level via pullup at start
  pinMode(SET_RELAYS2, INPUT);
  delay(500);
  ESP.restart();
}

// **************************************************************************
//  HTML Page for ESP System-Info
// **************************************************************************
//  ESP.getResetReason() returns a String containing the last reset reason in human readable format.
//  ESP.getFreeHeap() returns the free heap size.
//  ESP.getChipId() returns the ESP8266 chip ID as a 32-bit integer.
//  ESP.getCoreVersion() returns a String containing the core version.
//  ESP.getSdkVersion() returns the SDK version as a char.
//  ESP.getCpuFreqMHz() returns the CPU frequency in MHz as an unsigned 8-bit integer.
//  ESP.getSketchSize() returns the size of the current sketch as an unsigned 32-bit integer.
//  ESP.getFreeSketchSpace() returns the free sketch space as an unsigned 32-bit integer.
//  ESP.getSketchMD5() returns a lowercase String containing the MD5 of the current sketch.
//  ESP.getFlashChipId() returns the flash chip ID as a 32-bit integer.
//  ESP.getFlashChipSize() returns the flash chip size, in bytes, as seen by the SDK (may be less than actualsize).
//  ESP.getFlashChipRealSize() returns the real chip size, in bytes, based on the flash chip ID.
//  ESP.getFlashChipSpeed(void) returns the flash chip frequency, in Hz.
//  ESP.getCycleCount() returns the cpu instruction cycle count since start as an unsigned 32-bit. This is useful for accurate timing of very short actions like bit banging.
// *************************************************************************

void Handle_SysConf()
{
  
  String htmlSysConf;        // Hold the HTML Code
  buildHeader();
  buildFooter();
  htmlSysConf=htmlHeader;


  htmlSysConf+="        <div class='col-md-12'>\n";
  htmlSysConf+="          <h3>Master configuration</h3>\n";
  htmlSysConf+="          <table class='table table-striped' style='table-layout: fixed;'>\n";
  htmlSysConf+="            <thead>\n";
  htmlSysConf+="              <tr>\n";
  htmlSysConf+="                <th>Name</th>\n";
  htmlSysConf+="                <th>Value</th>\n";
  htmlSysConf+="              </tr>\n";
  htmlSysConf+="            </thead>\n";
  htmlSysConf+="            <tbody>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>Reset reason: </code></td>\n";
  htmlSysConf+="                <td><code>" + ESP.getResetReason() + " </code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>Chip ID: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getChipId()) + " </code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>Core version: </code></td>\n";
  htmlSysConf+="                <td><code>" + ESP.getCoreVersion() + " </code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>SDK version: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getSdkVersion()) + " </code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>Free heap: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getFreeHeap()) + " Byte</code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>CPU Frequency: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getCpuFreqMHz()) + " MHz </code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>Sketch size: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getSketchSize()) + " Byte </code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>Free sketch space: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getFreeSketchSpace()) + " Byte </code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>Sketch MD5: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getSketchMD5()) + " </code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>FlashChip ID: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getFlashChipId()) + " </code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>FlashChip size: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getFlashChipSize()) + " Byte</code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>FlashChip real size: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getFlashChipRealSize()) + " Byte</code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>FlashChip speed: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getFlashChipSpeed()) + " Hz</code></td></tr>\n";
  htmlSysConf+="              <tr class='text-uppercase'>\n";
  htmlSysConf+="                <td><code>CPU cycle count: </code></td>\n";
  htmlSysConf+="                <td><code>" + String(ESP.getCycleCount()) + " </code></td>\n";
  htmlSysConf+="              </tr>\n";
  htmlSysConf+="            </tbody>\n";
  htmlSysConf+="          </table>\n";
  htmlSysConf+="        </div>\n";
  htmlSysConf+="      </form>\n";
 
  htmlSysConf+=htmlFooter;

  httpServer.send(200, "text/html; charset=utf-8", htmlSysConf);
  httpServer.client().stop();

}


// **************************************************************************
//  HTML Page for ESP store Json config-file
// **************************************************************************
void Handle_config()
{
  if (httpServer.method() == HTTP_GET) {
    DEBUG_PRINTLN("WEB: Connection received - /config");
    sendConfigPage("", "", 0, 200);
  } else {
    DEBUG_PRINTLN("WEB: Connection received - /config (save)");
    sendConfigPage("Settings saved successfully! Please Reboot!", "Success!", 1, 200);
    }
}

//**************************************************************************
//  HTML Config-Page
//**************************************************************************

//---------------------------------------------------------------------
int Pick_Dec(const char * tx, int idx ) 
{
  int tmp = 0;
  
  for (int p = idx; p < idx + 5 && (tx[p] >= '0' && tx[p] <= '9') ; p++) {
    tmp = 10 * tmp + tx[p] - '0';
  }
  return tmp;
}
//----------------------------------------------------------------------------
int Pick_N_Zahl(const char * tx, char separator, byte n) 
{
  int ll = strlen(tx);
  int tmp = -1;
  byte anz = 1;
  byte i = 0;
  while (i < ll && anz < n) {
    if (tx[i] == separator)anz++;
    i++;
  }
  if (i < ll) return Pick_Dec(tx, i);
  else return -1;
}

//---------------------------------------------------------------------------
// Converts string from HTML input "01:23" to int minutes 83
int Time2Minutes(const char * timeString)
{
 return Pick_N_Zahl(timeString,':',1)*60 + Pick_N_Zahl(timeString,':',2);
}


void sendConfigPage(String message, String header, int type, int httpcode)
{
  char host_name_conf[20] = "";
  char mqtt_server_conf[30] = "";
  char mqtt_topic_conf[30] = "";
  char otau_server_conf[30] = "";
  char blockingtime_from_conf[6] = "00:00";
  char blockingtime_to_conf[6] = "23:59";
  char ntpserver_conf[30] = "";
  bool ntpenabled_conf;
  bool mqttenabled_conf;
  bool otauenabled_conf;
  int power_sampling_interval_conf;
  int bat_sampling_interval_conf;
  int sensors_count_conf;
  float adc_shift_conf;
  int saved_byte=0;

  if (type == 1){                                              // Type 1 -> save data
    String message = "WEB: Number of args received:";
    message += String(httpServer.args()) + "\n\r";
    for (int i = 0; i < httpServer.args(); i++) {
      message += "Arg " + (String)i + " --> ";
      message += httpServer.argName(i) + ":" ;
      message += httpServer.arg(i) + "\n\r";
    }
    if (httpServer.hasArg("ntpenabled")) {ntpenabled_conf = true;} else {ntpenabled_conf = false;}
    if (httpServer.hasArg("mqttenabled")) {mqttenabled_conf = true;} else {mqttenabled_conf = false;}
    if (httpServer.hasArg("otauenabled")) {otauenabled_conf = true;} else {otauenabled_conf = false;}
    strncpy(host_name_conf, httpServer.arg("host_name_conf").c_str(), 20);
    strncpy(mqtt_server_conf, httpServer.arg("mqtt_server_conf").c_str(), 30);
    strncpy(mqtt_topic_conf, httpServer.arg("mqtt_topic_conf").c_str(), 30);
    strncpy(ntpserver_conf, httpServer.arg("ntpserver_conf").c_str(), 30);
    strncpy(otau_server_conf, httpServer.arg("otau_server_conf").c_str(), 30);
    strncpy(blockingtime_from_conf, httpServer.arg("blockingtime_from_conf").c_str(), 6);
    strncpy(blockingtime_to_conf, httpServer.arg("blockingtime_to_conf").c_str(), 6);
    power_sampling_interval_conf = atoi(httpServer.arg("power_sampling_interval_conf").c_str());
    bat_sampling_interval_conf = atoi(httpServer.arg("bat_sampling_interval_conf").c_str());
    adc_shift_conf = atof(httpServer.arg("adc_shift_conf").c_str());
    sensors_count_conf = atoi(httpServer.arg("sensors_count_conf").c_str());

    // Update running system
    adc_shift = adc_shift_conf;                                               // Update adc_shift for main-page and battery measurement
    sensors_count = sensors_count_conf;                                       // Update sensors_count for Sensor-Config-Site
    otau_enabled = otauenabled_conf;
    mqtt_enabled = mqttenabled_conf;
    strncpy(otau_server,otau_server_conf,30);                                   
    
    DEBUG_PRINTLN(message);

    // validate values before saving
    bool validconf = true;
    // Check if begin less than end
    // if (Time2Minutes(blockingtime_from) < Time2Minutes(blockingtime_from)) {validconf = false;}
    
    if (validconf)
    {
      DEBUG_PRINTLN("SPI (sendConfigPage): save config.json...");
      DynamicJsonBuffer jsonBuffer;
      JsonObject& json = jsonBuffer.createObject();
      json["hostname"] = String(host_name_conf);
      json["mqttserver"] = String(mqtt_server_conf);
      json["mqtttopic"] = String(mqtt_topic_conf);
      json["mqttenabled"] = String(mqttenabled_conf);
      json["ntpserver"] = String(ntpserver_conf);
      json["ntpenabled"] = String(ntpenabled_conf);
      json["otauserver"] = String(otau_server_conf);
      json["otauenabled"] = String(otauenabled_conf);
      json["sampleintpower"] = String(power_sampling_interval_conf);
      json["sampleintbat"] = String(bat_sampling_interval_conf);
      json["sensorscount"] = String(sensors_count_conf);
      json["adc_shift"] = String(adc_shift_conf);
      json["blockingtimefrom"] = String(blockingtime_from_conf);
      json["blockingtimeto"] = String(blockingtime_to_conf);

      File configFile = LittleFS.open("/config.json", "w");
      if (!configFile) {
        DEBUG_PRINTLN("SPI (sendConfigPage): failed to open config file for writing");
      }
      #ifdef DEBUG
        json.printTo(Serial);DEBUG_PRINTLN("");
      #endif
      saved_byte = json.printTo(configFile);
      configFile.close();
      if (saved_byte < sizeof(jsonBuffer)) {
        DEBUG_PRINTLN("SPI (sendConfigPage): failed to save config file");
      } else {
        DEBUG_PRINT("SPI (sendConfigPage): config file saved: ");DEBUG_PRINTLN(saved_byte);
      }

      //end save
    } else {
      // not valid
//      type=3;                             // Type 3 -> date error -> check input
//      message="Check Inputdata!";
//      header="Error!";
    }  
  } else {                                // Type 0 -> load data
    if (LittleFS.begin())
    {
      DEBUG_PRINTLN("SPI (sendConfigPage): mounted file system");
      if (LittleFS.exists("/config.json"))
      {
        //file exists, reading and loading
        DEBUG_PRINTLN("SPI (sendConfigPage): reading config file");
        File configFile = LittleFS.open("/config.json", "r");
        if (configFile) {
          DEBUG_PRINT("SPI (sendConfigPage): opened config file - size [Byte]: ");
          size_t size = configFile.size();
          DEBUG_PRINTLN(size);
          // Allocate a buffer to store contents of the file.
          std::unique_ptr<char[]> buf(new char[size]);

          configFile.readBytes(buf.get(), size);
          DynamicJsonBuffer jsonBuffer;
          JsonObject& json = jsonBuffer.parseObject(buf.get());
          DEBUG_PRINT("JSO (sendConfigPage): ");
          #ifdef DEBUG
            json.printTo(Serial);
          #endif
          if (json.success()) {
            DEBUG_PRINTLN("\nJSO (sendConfigPage): parsed json");
            if (json.containsKey("hostname")) strncpy(host_name_conf, json["hostname"], 20);
            if (json.containsKey("mqttserver")) strncpy(mqtt_server_conf, json["mqttserver"], 30);
            if (json.containsKey("mqtttopic")) strncpy(mqtt_topic_conf, json["mqtttopic"], 30);
            if (json.containsKey("mqttenabled")) mqttenabled_conf = json["mqttenabled"];
            if (json.containsKey("ntpserver")) strncpy(ntpserver_conf, json["ntpserver"], 30);
            if (json.containsKey("ntpenabled")) ntpenabled_conf = json["ntpenabled"];
            if (json.containsKey("otauserver")) strncpy(otau_server_conf, json["otauserver"], 30);
            if (json.containsKey("otauenabled")) otauenabled_conf = json["otauenabled"];
            if (json.containsKey("blockingtimefrom")) strncpy(blockingtime_from_conf, json["blockingtimefrom"], 6);
            if (json.containsKey("blockingtimeto")) strncpy(blockingtime_to_conf, json["blockingtimeto"], 6);
            if (json.containsKey("sampleintpower")) power_sampling_interval_conf = json["sampleintpower"];
            if (json.containsKey("sampleintbat")) bat_sampling_interval_conf = json["sampleintbat"];
            if (json.containsKey("sensorscount")) sensors_count_conf = json["sensorscount"];
            if (json.containsKey("adc_shift")) adc_shift_conf = json["adc_shift"];
          } else {
            DEBUG_PRINTLN("JSO (sendConfigPage): failed to load json config");
          }
        }
      }
    } else {
      DEBUG_PRINTLN("SPI (sendConfigPage): failed to mount FS");
    }
  }
  String htmlDataconf;        // Hold the HTML Code
  buildHeader();
  buildFooter();
  htmlDataconf=htmlHeader;
  
  if (type == 1)
    htmlDataconf+="      <div class='row'><div class='col-md-12'><div class='alert alert-success'><strong>" + header + "!</strong> " + message + "</div></div></div>\n";
  if (type == 2)
    htmlDataconf+="      <div class='row'><div class='col-md-12'><div class='alert alert-warning'><strong>" + header + "!</strong> " + message + "</div></div></div>\n";
  if (type == 3)
    htmlDataconf+="      <div class='row'><div class='col-md-12'><div class='alert alert-danger'><strong>" + header + "!</strong> " + message + "</div></div></div>\n";
  htmlDataconf+="      <div class='row'>\n";
  htmlDataconf+="       <form method='post' action='/config'>";
  htmlDataconf+="        <div class='col-md-12'>\n";
  htmlDataconf+="          <h3>Configuration</h3>\n";
  htmlDataconf+="          <table class='table table-striped' style='table-layout: fixed;'>\n";
  htmlDataconf+="            <thead><tr><th>Option</th><th>Current Value</th><th>New Value</th></tr></thead>\n";
  htmlDataconf+="            <tbody>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>Hostname</td><td><code>" + ((host_name_conf[0] == 0 ) ? String("(" + String(host_name) + ")") : String(host_name_conf)) + "</code></td><td><input type='text' id='host_name_conf' name='host_name_conf' value='" + String(host_name_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>MQTT Server</td><td><code>" + String(mqtt_server_conf) + "</code></td><td><input type='text' id='mqtt_server_conf' name='mqtt_server_conf' value='" + String(mqtt_server_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>MQTT Topic</td><td><code>" + String(mqtt_topic_conf) + "</code></td><td><input type='text' id='mqtt_topic_conf' name='mqtt_topic_conf' value='" + String(mqtt_topic_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>MQTT enabled?</td><td><code>" + (mqttenabled_conf ? String("Yes") : String("No")) + "</code></td><td><input type='checkbox' id='mqttena' name='mqttenabled' " + (mqttenabled_conf ? String("checked") : String("")) + "></td></tr>";
  htmlDataconf+="            <tr class='text-uppercase'><td>NTP Server</td><td><code>" + ((ntpserver_conf[0] == 0 ) ? String("(" + String(poolServerName) + ")") : String(ntpserver_conf)) + "</code></td><td><input type='text' id='ntpserver_conf' name='ntpserver_conf' value='" + String(ntpserver_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>NTP enabled?</td><td><code>" + (ntpenabled_conf ? String("Yes") : String("No")) + "</code></td><td><input type='checkbox' id='ntpena' name='ntpenabled' " + (ntpenabled_conf ? String("checked") : String("")) + "></td></tr>";
  htmlDataconf+="            <tr class='text-uppercase'><td>OTA-Update Server</td><td><code>" + String(otau_server_conf) + "</code></td><td><input type='text' id='otau_server_conf' name='otau_server_conf' value='" + String(otau_server_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>OTA-Update enabled?</td><td><code>" + (otauenabled_conf ? String("Yes") : String("No")) + "</code></td><td><input type='checkbox' id='otauena' name='otauenabled' " + (otauenabled_conf ? String("checked") : String("")) + "></td></tr>";
  htmlDataconf+="            <tr class='text-uppercase'><td>Measure Freq Power Mode (minutes)</td><td><code>" + String(power_sampling_interval_conf) + "</code></td><td><input type='text' id='power_sampling_interval_conf' name='power_sampling_interval_conf' value='" + String(power_sampling_interval_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>Measure Freq Battery Mode (minutes)</td><td><code>" + String(bat_sampling_interval_conf) + "</code></td><td><input type='text' id='bat_sampling_interval_conf' name='bat_sampling_interval_conf' value='" + String(bat_sampling_interval_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>Number of Sensors</td><td><code>" + String(sensors_count_conf) + "</code></td><td><select name='sensors_count_conf' size='1'><option " + ((sensors_count_conf == 1 ) ? String("selected") : String("")) + ">1</option><option "+ ((sensors_count_conf == 2 ) ? String("selected") : String("")) +">2</option><option "+ ((sensors_count_conf == 3 ) ? String("selected") : String("")) +">3</option><option "+ ((sensors_count_conf == 4 ) ? String("selected") : String("")) +">4</option><option "+ ((sensors_count_conf == 5 ) ? String("selected") : String("")) +">5</option><option "+ ((sensors_count_conf == 6 ) ? String("selected") : String("")) +">6</option><option "+ ((sensors_count_conf == 7 ) ? String("selected") : String("")) +">7</option><option "+ ((sensors_count_conf == 8 ) ? String("selected") : String("")) +">8</option><option "+ ((sensors_count_conf == 9 ) ? String("selected") : String("")) +">9</option><option "+ ((sensors_count_conf == 10 ) ? String("selected") : String("")) +">10</option></select></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>Battery voltage correction</td><td><code>" + String(adc_shift_conf) + "</code></td><td><input type='text' id='adc_shift_conf' name='adc_shift_conf' value='" + String(adc_shift_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>Blocking time from</td><td><code>" + String(blockingtime_from_conf) + "</code></td><td><input type='time' id='blockingtime_from_conf' name='blockingtime_from_conf' min='00:00' max='23:59' required pattern='[0-9]{2}:[0-9]{2}' value='" + String(blockingtime_from_conf) + "'></td></tr>\n";
  htmlDataconf+="            <tr class='text-uppercase'><td>Blocking time to</td><td><code>" + String(blockingtime_to_conf) + "</code></td><td><input type='time' id='blockingtime_to_conf' name='blockingtime_to_conf'  min='00:00' max='23:59' required pattern='[0-9]{2}:[0-9]{2}' value='" + String(blockingtime_to_conf) + "' ></td></tr>\n";
  htmlDataconf+="            <tr><td colspan='5' class='text-center'><em><a href='/reboot' class='btn btn-sm btn-danger'>Reboot</a>  <a href='/update' class='btn btn-sm btn-warning'>Update</a>  <button type='submit' class='btn btn-sm btn-success'>Save</button> <a href='/sensor_config' class='btn btn-sm btn-info'>Sensors</a>  <a href='/' class='btn btn-sm btn-primary'>Cancel</a>";
  if (wakeup)
    htmlDataconf+="          <a href='/gosleep' class='btn btn-sm btn-light'>Go sleeping</a>";  
  htmlDataconf+="            </em></td></tr>";
  htmlDataconf+="            </tbody></table>\n";
  htmlDataconf+="          </div></div>\n";
  
  htmlDataconf+=htmlFooter;

  httpServer.send(httpcode, "text/html; charset=utf-8", htmlDataconf);
  httpServer.client().stop();
}

// **************************************************************************
//  HTML Page for ESP store Json sensor-config-file
// **************************************************************************
void Handle_sensor_config()
{
  if (httpServer.method() == HTTP_GET) {
    DEBUG_PRINTLN("WEB: Connection received - /sensor_config");
    sendSensorConfigPage("", "", 0, 200);
  } else {
    DEBUG_PRINTLN("WEB: Connection received - /sensor_config (save)");
    sendSensorConfigPage("Sensor settings saved successfully!", "Success!", 1, 200);
    sendMQTT(MQTT_TRESHOLD);
  }
}

// **************************************************************************
//  //  HTML Sensor-Config-Page
// **************************************************************************
String setBackgroundColor(int i, int mois_treshold_conf)
// #EFE78B - yellow -> dry
// #8ADBF2 - blue - wet
{
  return (freqValue[i] > mois_treshold_conf) ? "#EFE78B" : "#8ADBF2";
}

void sendSensorConfigPage(String message, String header, int type, int httpcode)
{
  char sensor_index[][3] = {"00","01","02","03","04","05","06","07","08","09","10"};
  char sensor_name_conf[10][25];
  char jsonSensorName[20];
  bool sensorenabled_conf[10];
  char jsonSensorEnabled[20];
  int sensorgroup_conf[10];
  char jsonSensorGroup[20];
  int average_weight_conf[10];
  char jsonAverageWeight[20];
  int mois_treshold_conf[10];
  char jsonMoisTreshold[20];
  char httpServerArg[20];
  float tresholdGroup1_conf=0.0;
  float tresholdGroup2_conf=0.0;
  
  for (int i = 1; i <= sensors_count ; i++) {
    strcpy(sensor_name_conf[i], ""); 
  }
  
  if (type == 1){                                              // Type 1 -> save data
    String message = "WEB: Number of args received:";
    message += String(httpServer.args()) + "\n\r";
    for (int i = 0; i < httpServer.args(); i++) {
      message += "Arg " + (String)i + " --> ";
      message += httpServer.argName(i) + ":" ;
      message += httpServer.arg(i) + "\n\r";
    }
    for (int i = 1; i <= sensors_count ; i++) {
      strcpy(httpServerArg, "sensor_name");
      strcat(httpServerArg, sensor_index[i]);
      strcat(httpServerArg, "_conf");
      strcpy(sensor_name_conf[i], httpServer.arg(httpServerArg).c_str());
      strcpy(sensorName[i], sensor_name_conf[i]);

      strcpy(httpServerArg, "sensorenabled");
      strcat(httpServerArg, sensor_index[i]);
      strcat(httpServerArg, "_conf");
      if (httpServer.hasArg(httpServerArg)) {
        sensorenabled_conf[i] = true;
        sensorAvailable[i] = true;
      } else {
        sensorenabled_conf[i] = false;
        sensorAvailable[i] = false;
      }

      strcpy(httpServerArg, "sensorgroup");
      strcat(httpServerArg, sensor_index[i]);
      strcat(httpServerArg, "_conf");
      sensorgroup_conf[i] = atoi(httpServer.arg(httpServerArg).c_str());
      sensorGroup[i] = sensorgroup_conf[i];
      
      strcpy(httpServerArg, "average_weight");
      strcat(httpServerArg, sensor_index[i]);
      strcat(httpServerArg, "_conf");
      average_weight_conf[i] = atoi(httpServer.arg(httpServerArg).c_str());
      sensorWeighting[i] = average_weight_conf[i];

      strcpy(httpServerArg, "mois_treshold");
      strcat(httpServerArg, sensor_index[i]);
      strcat(httpServerArg, "_conf");
      mois_treshold_conf[i] = atoi(httpServer.arg(httpServerArg).c_str());
      sensorTreshold[i] = mois_treshold_conf[i];
    }

    tresholdGroup1_conf = atof(httpServer.arg("tresholdGroup1_conf").c_str());
    tresholdGroup1 = tresholdGroup1_conf;

    tresholdGroup2_conf = atof(httpServer.arg("tresholdGroup2_conf").c_str());
    tresholdGroup2 = tresholdGroup2_conf;

    DEBUG_PRINTLN(message);

    // validate values before saving
    bool validconf = true;
    if (validconf)
    {
      DEBUG_PRINTLN("SPI (sendSensorConfigPage): save sensor_config.json...");
      DynamicJsonBuffer jsonBuffer;
      JsonObject& json = jsonBuffer.createObject();
      for (int i = 1; i <= sensors_count ; i++) {
        //sprintf (sensor_index, "%02d", i);

        strcpy(jsonSensorName, "sensor_name");
        strcat(jsonSensorName, sensor_index[i]);
        json[jsonSensorName] = String(sensor_name_conf[i]);
        strcpy(jsonSensorEnabled, "sensorenabled");
        strcat(jsonSensorEnabled, sensor_index[i]);
        json[jsonSensorEnabled] = String(sensorenabled_conf[i]);
        strcpy(jsonSensorGroup, "sensorgroup");
        strcat(jsonSensorGroup, sensor_index[i]);
        json[jsonSensorGroup] = String(sensorgroup_conf[i]);
        strcpy(jsonAverageWeight, "average_weight");
        strcat(jsonAverageWeight, sensor_index[i]);
        json[jsonAverageWeight] = String(average_weight_conf[i]);
        strcpy(jsonMoisTreshold, "mois_treshold");
        strcat(jsonMoisTreshold, sensor_index[i]);
        json[jsonMoisTreshold] = String(mois_treshold_conf[i]);
      }  
      json["tresholdGroup1"] = String(tresholdGroup1_conf);
      json["tresholdGroup2"] = String(tresholdGroup2_conf);


      File configFile = LittleFS.open("/sensor_config.json", "w");
      if (!configFile) {
        DEBUG_PRINTLN("SPI (sendSensorConfigPage): failed to open sensor_config file for writing");
      }
      #ifdef DEBUG
        json.printTo(Serial);
      #endif
      DEBUG_PRINTLN("");
      json.printTo(configFile);
      configFile.close();
      //end save
    }
  } else {                                // Type 0 -> load data
    if (LittleFS.begin())
    {
      DEBUG_PRINTLN("SPI (sendSensorConfigPage): mounted file system");
      if (LittleFS.exists("/sensor_config.json"))
      {
        //file exists, reading and loading
        DEBUG_PRINTLN("SPI (sendSensorConfigPage): reading sensor_config file");
        File configFile = LittleFS.open("/sensor_config.json", "r");
        if (configFile) {
          DEBUG_PRINTLN("SPI: opened sensor_config file");
          size_t size = configFile.size();
          // Allocate a buffer to store contents of the file.
          std::unique_ptr<char[]> buf(new char[size]);

          configFile.readBytes(buf.get(), size);
          DynamicJsonBuffer jsonBuffer;
          JsonObject& json = jsonBuffer.parseObject(buf.get());
          DEBUG_PRINT("JSO (sendSensorConfigPage): ");
          #ifdef DEBUG
            json.printTo(Serial);
          #endif
          if (json.success()) {
            DEBUG_PRINTLN("\nJSO (sendSensorConfigPage): parsed json");
            for (int i = 1; i <= sensors_count ; i++) {
              //sprintf (sensor_index, "%02d", i);
              DEBUG_PRINT("Read Sensor_Index: ");
              DEBUG_PRINTLN(sensor_index[i]);

              strcpy(jsonSensorName, "sensor_name");
              strcat(jsonSensorName, sensor_index[i]);
              if (json.containsKey(jsonSensorName)) strncpy(sensor_name_conf[i], json[jsonSensorName],20);
              strcpy(jsonSensorEnabled, "sensorenabled");
              strcat(jsonSensorEnabled, sensor_index[i]);
              if (json.containsKey(jsonSensorEnabled)) sensorenabled_conf[i] = json[jsonSensorEnabled];
              strcpy(jsonSensorGroup, "sensorgroup");
              strcat(jsonSensorGroup, sensor_index[i]);
              if (json.containsKey(jsonSensorGroup)) sensorgroup_conf[i] = json[jsonSensorGroup];
              strcpy(jsonAverageWeight, "average_weight");
              strcat(jsonAverageWeight, sensor_index[i]);
              if (json.containsKey(jsonAverageWeight)) average_weight_conf[i] = json[jsonAverageWeight];
              strcpy(jsonMoisTreshold, "mois_treshold");
              strcat(jsonMoisTreshold, sensor_index[i]);
              if (json.containsKey(jsonMoisTreshold)) mois_treshold_conf[i] = json[jsonMoisTreshold];
            }  
            if (json.containsKey("tresholdGroup1")) tresholdGroup1_conf = json["tresholdGroup1"];
            if (json.containsKey("tresholdGroup2")) tresholdGroup2_conf = json["tresholdGroup2"];
          } else {
              DEBUG_PRINTLN("JSO (sendSensorConfigPage): failed to load json sensor_config");
          }
        }
      }
    } else {
      DEBUG_PRINTLN("SPI (sendSensorConfigPage): failed to mount FS");
    }
  }
  
//-------------------- 
  outputMoisture();            // Calculate treshold before page view
  // Split htmlSensorconf in tree parts becouse TCP buffer overflow
  String htmlSensorconf_1;        // Hold the HTML Code Part 1
  String htmlSensorconf_2;        // Hold the HTML Code Part 2
  String htmlSensorconf_3;        // Hold the HTML Code Part 3
  buildHeader();
  buildFooter();
  htmlSensorconf_1=htmlHeader;
  
  if (type == 1)
    htmlSensorconf_1+="      <div class='row'><div class='col-md-12'><div class='alert alert-success'><strong>" + header + "!</strong> " + message + "</div></div></div>\n";
  if (type == 2)
    htmlSensorconf_1+="      <div class='row'><div class='col-md-12'><div class='alert alert-warning'><strong>" + header + "!</strong> " + message + "</div></div></div>\n";
  if (type == 3)
    htmlSensorconf_1+="      <div class='row'><div class='col-md-12'><div class='alert alert-danger'><strong>" + header + "!</strong> " + message + "</div></div></div>\n";
  htmlSensorconf_1+="      <div class='row'>\n";
  htmlSensorconf_1+="      <form method='post' action='/sensor_config'>\n";
  htmlSensorconf_1+="        <div class='col-md-12'>\n";
  htmlSensorconf_1+="          <h3>Sensor configuration</h3>\n";
  htmlSensorconf_1+="          <table class='table table-striped' style='table-layout: fixed;'>\n";
  htmlSensorconf_1+="            <thead>\n";
  htmlSensorconf_1+="              <tr>\n";
  htmlSensorconf_1+="                <th>Sensorname</th>\n";
  htmlSensorconf_1+="                <th>Value</th>\n";
  htmlSensorconf_1+="                <th>Battery voltage</th>\n";
  htmlSensorconf_1+="                <th>Sensor enabled?</th>\n";
  htmlSensorconf_1+="                <th>Sensor group</th>\n";
  htmlSensorconf_1+="                <th>average weighting</th>\n";
  htmlSensorconf_1+="                <th>moisture threshold</th>\n";
  htmlSensorconf_1+="              </tr>\n";
  htmlSensorconf_1+="            </thead>\n";
  htmlSensorconf_1+="            <tbody>\n";

  for (int i = 1; i <= sensors_count ; i++) {
    htmlSensorconf_2+="              <tr class='text-uppercase'>\n";
    htmlSensorconf_2+="                <td><input id='sensor_name" + String(sensor_index[i]) + "_conf' name='sensor_name" + String(sensor_index[i]) + "_conf' value='" + String(sensor_name_conf[i]) + "' type='text' maxlength='25' size='13'></td>\n";
    htmlSensorconf_2+="                <td><code>" + String(freqValue[i]) + " kHz" + ((sensorAvailable[i]) ? String("") : String(" *)")) + "</code></td>\n";
    htmlSensorconf_2+="                <td><code>" + String(analogValue[i]) + " V</code></td>\n";
    htmlSensorconf_2+="                <td><input id='sensorenabled" + String(sensor_index[i]) + "_conf' name='sensorenabled" + String(sensor_index[i]) + "_conf' " + (sensorenabled_conf[i] ? String("checked") : String("")) + " type='checkbox'></td>\n";
    htmlSensorconf_2+="                <td>\n";
    htmlSensorconf_2+="                    <input id='group1_" + String(sensor_index[i]) + "_conf' value='1' name='sensorgroup" + String(sensor_index[i]) + "_conf' " + ((sensorgroup_conf[i] == 1) ? String("checked") : String(""))  + " type='radio'><label for='group1_" + String(sensor_index[i]) + "_conf'>1</label>\n"; 
    htmlSensorconf_2+="                    <input id='group2_" + String(sensor_index[i]) + "_conf' value='2' name='sensorgroup" + String(sensor_index[i]) + "_conf' " + ((sensorgroup_conf[i] == 2) ? String("checked") : String(""))  + " type='radio'><label for='group2_" + String(sensor_index[i]) + "_conf'>2</label>\n";
    htmlSensorconf_2+="                </td>\n";
    htmlSensorconf_2+="                <td><input id='average_weight" + String(sensor_index[i]) + "_conf' name='average_weight" + String(sensor_index[i]) + "_conf' value='" + String(average_weight_conf[i]) + "' maxlength='3' size='5' pattern='[0-9]+' style='text-align: right;' type='text'> %</td>\n";
    htmlSensorconf_2+="                <td><input id='mois_treshold" +  String(sensor_index[i]) + "_conf' name='mois_treshold" + String(sensor_index[i]) + "_conf'  value='" + String(mois_treshold_conf[i]) +  "' maxlength='3' size='5' pattern='[0-9]+' style='text-align: right; background-color: " + setBackgroundColor(i, mois_treshold_conf[i]) + ";' type='text'></td>\n";
    htmlSensorconf_2+="              </tr>\n";
  }
  htmlSensorconf_3+="              <tr>\n";
  htmlSensorconf_3+="                <td colspan='3'>*) Sensors currently not available</td>\n";
//  htmlSensorconf_3+="                <td>&nbsp;</td>\n";
//  htmlSensorconf_3+="                <td>&nbsp;</td>\n";
  htmlSensorconf_3+="                <td colspan='2'>Averadge Group 1: " + String(rtcMem.averageTresholdValueGroup1) + "</td>\n";
  htmlSensorconf_3+="                <td colspan='2'>Averadge Group 2: " + String(rtcMem.averageTresholdValueGroup2) + "</td>\n";
  htmlSensorconf_3+="              </tr>\n";

  htmlSensorconf_3+="              <tr>\n";
  htmlSensorconf_3+="                <td>&nbsp;</td>\n";
  htmlSensorconf_3+="                <td>&nbsp;</td>\n";
  htmlSensorconf_3+="                <td>&nbsp;</td>\n";
  htmlSensorconf_3+="                <td>Treshold Group 1:</td>\n";
  htmlSensorconf_3+="                <td><output name='trasholdOutputNameGroup1' id='tresholdGroup1_output'>" + String(tresholdGroup1_conf) + "</output></td>\n";
  htmlSensorconf_3+="                <td>Treshold Group 2:</td>\n";
  htmlSensorconf_3+="                <td><output name='trasholdOutputNameGroup2' id='tresholdGroup2_output'>" + String(tresholdGroup2_conf) + "</output></td>\n";
  htmlSensorconf_3+="              </tr>\n";

  htmlSensorconf_3+="              <tr>\n";
  htmlSensorconf_3+="                <td>&nbsp;</td>\n";
  htmlSensorconf_3+="                <td>&nbsp;</td>\n";
  htmlSensorconf_3+="                <td>&nbsp;</td>\n";
  htmlSensorconf_3+="                <td colspan='2'> <input type=range id='tresholdGroup1_conf' name='tresholdGroup1_conf' value=" + String(tresholdGroup1_conf) + " max='1' min='0' step='0.1' oninput='tresholdGroup1_output.value = tresholdGroup1_conf.value'></td>\n";
  htmlSensorconf_3+="                <td colspan='2'> <input type=range id='tresholdGroup2_conf' name='tresholdGroup2_conf' value=" + String(tresholdGroup2_conf) + " max='1' min='0' step='0.1' oninput='tresholdGroup2_output.value = tresholdGroup2_conf.value'></td>\n"; 
  htmlSensorconf_3+="              </tr>\n";
  
  htmlSensorconf_3+="              <tr>\n";
  htmlSensorconf_3+="                <td colspan='6' class='text-center'><em><a href='/reboot' class='btn btn-sm btn-danger'>Reboot</a>\n";
  htmlSensorconf_3+="                    <a href='/update' class='btn btn-sm btn-warning'>Update</a>\n";
  htmlSensorconf_3+="                    <button type='submit' class='btn btn-sm btn-success'>Save</button>\n";
  htmlSensorconf_3+="                    <a href='/config' class='btn btn-sm btn-info'>Conguration</a>\n";
  htmlSensorconf_3+="                    <a href='/' class='btn btn-sm btn-primary'>Cancel</a></em>\n";
  htmlSensorconf_3+="                </td>\n";
  htmlSensorconf_3+="              </tr>\n";
  htmlSensorconf_3+="            </tbody>\n";
  htmlSensorconf_3+="          </table>\n";
  htmlSensorconf_3+="        </div>\n";
  htmlSensorconf_3+="      </form>\n";
  htmlSensorconf_3+="    </div>\n";
      
  htmlSensorconf_3+=htmlFooter;

//  DEBUG_PRINTLN(htmlSensorconf);
  httpServer.setContentLength(htmlSensorconf_1.length() + htmlSensorconf_2.length() + htmlSensorconf_3.length());  
  httpServer.send(200,"text/html; charset=utf-8",htmlSensorconf_1);        
  httpServer.sendContent(htmlSensorconf_2);                 
  httpServer.sendContent(htmlSensorconf_3);                 
//  httpServer.send(200, "text/html; charset=utf-8", htmlSensorconf);
  httpServer.client().stop();
}


// **************************************************************************
//  HTML get Value Page
// **************************************************************************
void Handle_getvalue(){
  String htmlDataconf;        // Hold the HTML Code
  buildHeader();
  buildFooter();
  htmlDataconf=htmlHeader;
  int i;

  htmlDataconf+="Battery Master: <b>" + String(analogValue[0]) + "V</b><br><br>";               // Battery Voltage Master
  htmlDataconf+="<hr />\n";
  
  for (i = 1; i <= sensors_count ; i++) {
    htmlDataconf+="Gies-O-Mat-Sensor " + String(i) + ": " + String(freqValue[i]) + " --- ";      // MQTT-Abfrage
    htmlDataconf+="Battery :" + String(analogValue[i]) + "V - " + arrivingTime[i] + "<br>\n";   // Battery Voltage Sensors
//    if ((i+1) < sensors_count)
//      htmlDataconf+=";";
  }

  htmlDataconf+=htmlFooter;
  
  httpServer.send(200, "text/html; charset=utf-8", htmlDataconf);
  httpServer.client().stop(); 
}


// **************************************************************************
//  HTML Main Page
// **************************************************************************
void Handle_welcome(){
 
  String htmlDataMain;        // Hold the HTML Code
  buildHeader();
  buildFooter();
  htmlDataMain=htmlHeader;
  
  htmlDataMain+="        <div class='col-md-12'>\n";
  htmlDataMain+="          <h3>Plant Soil Moisture Master</h3><br>";

  htmlDataMain+="             Battery Voltage Master: <b>" + String(analogValue[0]) + " V</b><br>\n";
  htmlDataMain+="             Treshold calculation: " + ((disableTresholdCalculation) ? String("<b>off</b><br>\n") : String("<b>on</b><br>\n"));
  htmlDataMain+="             Moisture-Status Sensor Channel 1: " + ((relays1_state) ? String("<b>wet</b><br>\n") : String("<b>dry</b><br>\n")); 
  if (!enableHumidCalculationSensor1){
     htmlDataMain+="          (Humid calculation for sensor input 1 disabled by topic - set status to wet)<br>\n";
  }
  htmlDataMain+="             Moisture-Status Sensor Channel 2: " + ((relays2_state) ? String("<b>wet</b><br>\n") : String("<b>dry</b><br>\n")); 
  if (!enableHumidCalculationSensor2){
     htmlDataMain+="          (Humid calculation for sensor input 2 disabled by topic - set status to wet)<br>\n";
  }

  htmlDataMain+="<hr />\n";
  
  for (int i = 1; i <= sensors_count ; i++) {
     if (sensorAvailable[i]) {
        htmlDataMain+= String(sensorName[i]) + ": Battery Voltage <b>" + String(analogValue[i]) + " V </b>";
        htmlDataMain+="Last Value <b>" + String(freqValue[i]) + " kHz </b>- " + arrivingTime[i] +"<br>\n";
     } else {
        htmlDataMain+=String(sensorName[i]) + "<b> disabled</b><br>\n";
     }
  }
  htmlDataMain+="           <br><b>Possible URL Prefix:</b><br>\n";
  htmlDataMain+="           /reboot<br>\n";
  htmlDataMain+="           /gosleep<br>";
  htmlDataMain+="           /clearconfig<br>\n";
  htmlDataMain+="           /freemem<br>\n";
  htmlDataMain+="           /getvalue<br>\n";
  htmlDataMain+="           /config<br>\n";
  htmlDataMain+="           /sensor_config<br>\n";
  htmlDataMain+="           /sysconf<br><br>\n";
  htmlDataMain+="          </div><br>\n";

  htmlDataMain+=htmlFooter;

  httpServer.send(200, "text/html; charset=utf-8", htmlDataMain);
  httpServer.client().stop();  
}

// **************************************************************************
//  MQTT Recieve subscribing Frequency and Battery Value from MQTT Broker
// **************************************************************************
// C substring function definition
void substring(char s[], char sub[], int p, int l) {
   int c = 0;
   
   while (c < l) {
      sub[c] = s[p+c-1];
      c++;
   }
   sub[c] = '\0';
}

void callbackMQTT(char* topic, byte* payload, unsigned int length) {
  const char mois[] = "moisture";
  const char batt[] = "battery";
  const char cbWakeup[] = "wakeup";
  const char cHumidCalc1[] = "enaHumidCalcSensor1";
  const char cHumidCalc2[] = "enaHumidCalcSensor2";
  const char cLastTransmission[] = "last-transmission";
  tmElements_t tLastTransmission;
  int Year, Month, Day, Hour, Minute, Second ;
  
  // Hilfsvariablen für die Konvertierung der Nachricht in ein String
  char message_buff[100]; 
  char sensornr[3];
  char *ptr;
  int i;

  DEBUG_PRINT("Message arrived [");
  DEBUG_PRINT(topic);
  DEBUG_PRINT("] ");
  for (i=0; i<length; i++) {
    DEBUG_PRINT((char)payload[i]);
    message_buff[i] =(char)payload[i];
  }
  message_buff[i+1]='\0';
  DEBUG_PRINTLN();

  // kopiere 2 letzte Zeichen von topic (Sensornummer)
  substring(topic, sensornr, strlen(topic)-1, 2);
  
  DEBUG_PRINT("SensorNr: ");
  DEBUG_PRINTLN(sensornr);

  // Ende-Zeichen setzen
  sensornr[3] = '\0';

  // do not receive undefined sensors 
  if (atoi(sensornr) <= sensors_count) {
    // receive last-transmission per sensor
    // If the last value is older than 6 h, deactivate the sensor for the calculation. 
    ptr = strstr(topic, cLastTransmission);
    if ((ptr) && (atoi(sensornr) >= 1))  { 
       sscanf(message_buff, "%04d-%02d-%02d %02d:%02d:%02d", &Year, &Month, &Day, &Hour, &Minute, &Second);
       tLastTransmission.Year = Year - 1970;
       tLastTransmission.Month = Month;
       tLastTransmission.Day = Day;
       tLastTransmission.Hour = Hour;
       tLastTransmission.Minute = Minute;
       tLastTransmission.Second = Second;
       
       sprintf(arrivingTime[atoi(sensornr)], "%04d-%02d-%02d %02d:%02d:%02d ", year(makeTime(tLastTransmission)), month(makeTime(tLastTransmission)), day(makeTime(tLastTransmission)), hour(makeTime(tLastTransmission)), minute(makeTime(tLastTransmission)), second(makeTime(tLastTransmission)));   // put arriving time per sense

       DEBUG_PRINT("MQTT: Receive LastTransmission: ");
       DEBUG_PRINTLN(arrivingTime[atoi(sensornr)]);
       // test if sensor has responded in the last 6 hours
       DEBUG_PRINT("MQTT: Sensor available?: ");
       // difftime returns time2 - time1 in seconds
       if (difftime(now(), makeTime(tLastTransmission)) > 6*60*60 ) {
         sensorAvailable[atoi(sensornr)] = false;
         DEBUG_PRINTLN("no");
       } else {          
         sensorAvailable[atoi(sensornr)] = true;
         DEBUG_PRINTLN("yes");
       }
    }   
   
    ptr = strstr(topic, mois);
    if ((ptr) && (atoi(sensornr) >= 1))  {
      freqValue[atoi(sensornr)] = atof(message_buff);            // Sensor Frequency (max. 10 Slaves)
      DEBUG_PRINT("MQTT: Receive moisture value: ");
      DEBUG_PRINTLN(freqValue[atoi(sensornr)]);
      receivedFreqValue[atoi(sensornr)] = true;                  // true for received data
      outputMoisture();                                          // Calculate an set dry/wet-Status to Gardena Computer
      getRelayState();                                           // Check relays status
    }

    ptr = strstr(topic, batt);
    if ((ptr) && (atoi(sensornr) >= 1))  {
      analogValue[atoi(sensornr)] = atof(message_buff);          // Sensor Battery Status (max. 10 Slaves)
    }

    // receive wakeup topic for master?
    ptr = strstr(topic, cbWakeup);
    if ((ptr) && (atoi(sensornr) == 0) && !powermode && !wakeup)    {
      (strncmp(message_buff, "1", 1) == 0 ) ? wakeup = true : wakeup = false;                                // Master go sleep? 
      DEBUG_PRINT("MQTT: Receive wakeup flag: ");
      DEBUG_PRINTLN(wakeup);
      enableWebService();
      sendMQTT(MQTT_IMAWAKE); // send "I am awake"
    } 

    // receive enable/disable humid calculation for Sensor 1
    ptr = strstr(topic, cHumidCalc1);
    if ((ptr) && (atoi(sensornr) == 0))  {
      (strncmp(message_buff, "1", 1) == 0 ) ? enableHumidCalculationSensor1 = true : enableHumidCalculationSensor1 = false;  // Enable humid calculation? 
      DEBUG_PRINT("MQTT: Receive enable/disable humid calculation for Sensor 1: ");
      DEBUG_PRINTLN(enableHumidCalculationSensor1);
      outputMoisture(); // Enable/disable humid calculation and recalc wet/dry-state
    }

    // receive enable/disable humid calculation for Sensor 2
    ptr = strstr(topic, cHumidCalc2);
    if ((ptr) && (atoi(sensornr) == 0))  {
      (strncmp(message_buff, "1", 1) == 0 ) ? enableHumidCalculationSensor2 = true : enableHumidCalculationSensor2 = false;  // Enable humid calculation? 
      DEBUG_PRINT("MQTT: Receive enable/disable humid calculation for Sensor 2: ");
      DEBUG_PRINTLN(enableHumidCalculationSensor2);
      outputMoisture(); // Enable/disable humid calculation and recalc wet/dry-state
    }
  } else {
    DEBUG_PRINT("MQTT: Sensor ");  
    DEBUG_PRINT(sensornr);  
    DEBUG_PRINTLN(" not configured!");  
  } // end undefined sensors
}

// **************************************************************************
//  MQTT Publish Battery Value Master to MQTT Broker
// **************************************************************************
void sendMQTT(int sendTopic) {
  bool mqttresult;
  char mqtt_battery_topic[40] = "";
  char mqtt_treshold_topic[40] = "";
  char mqtt_wet_state_topic[40] = "";
  char mqtt_averageTresholdValueGroup_topic[40] = "";
  char mqtt_version_topic[40] = "";
  char mqtt_last_transmission_topic[40] = "";
  char mqtt_wakeup_topic[40] = "";
  char mqtt_iamawake_topic[40] = "";
  char mqtt_error_topic[40] = "";
  char nowTime[25];
  char sensornumber[3] = "";

  sprintf(nowTime, "%04d-%02d-%02d %02d:%02d:%02d ", year(now()), month(now()), day(now()), hour(now()), minute(now()), second(now()));
  
  if ((mqtt_enabled) && client.connected()){
    DEBUG_PRINT("MQTT: Enter sendMQTT(");
    DEBUG_PRINT(sendTopic);
    DEBUG_PRINTLN(")");

    // Publish Battery Value
    if (sendTopic & MQTT_BATTERY) {
        strcpy (mqtt_battery_topic,mqtt_topic);
        strcat (mqtt_battery_topic,"/battery/master/00");
#ifdef MQTT_PUBLISH
        mqttresult = client.publish(mqtt_battery_topic, String(analogValue[0]).c_str(), true);
        if (!mqttresult) {
          DEBUG_PRINT("MQTT: battery value publish failed -> rc: ");
          DEBUG_PRINTLN(mqttresult);
        } else {
          DEBUG_PRINTLN("MQTT: battery value publish successful ...");
        }
#else
        DEBUG_PRINTLN("MQTT: battery value publish disabled ...");
#endif
    } 

    // Publish treshold values
    if (sendTopic & MQTT_TRESHOLD) {
      for (int i = 1; i <= sensors_count ; i++) {
        strcpy (mqtt_treshold_topic,mqtt_topic);
        strcat (mqtt_treshold_topic,"/treshold/sensor/");
        sprintf(sensornumber, "%02d", i);
        strcat (mqtt_treshold_topic,sensornumber);
#ifdef MQTT_PUBLISH
        mqttresult = client.publish(mqtt_treshold_topic, String(sensorTreshold[i]).c_str(), true);  
        if (!mqttresult) {
          DEBUG_PRINT("MQTT: current TRESHOLD Sensor ");
          DEBUG_PRINT(i); 
          DEBUG_PRINT( "publish failed -> rc: ");
          DEBUG_PRINTLN(mqttresult);
        } else {
          DEBUG_PRINT("MQTT: current TRESHOLD Sensor ");
          DEBUG_PRINT(i);
          DEBUG_PRINT(" publish successful -> ");
          DEBUG_PRINTLN(String(sensorTreshold[i]).c_str());
        }
#else
          DEBUG_PRINT("MQTT: current TRESHOLD Sensor ");
          DEBUG_PRINT(i);
          DEBUG_PRINTLN(" publish disabled ...");
#endif
      }
    }  

    // Publish wet state sensor 1
    if (sendTopic & MQTT_STATE) {
      strcpy (mqtt_wet_state_topic,mqtt_topic);
      strcat (mqtt_wet_state_topic,"/humid/master/state/1");
#ifdef MQTT_PUBLISH
      mqttresult = client.publish(mqtt_wet_state_topic, String(!relays1_state).c_str(), true);  // 0=dry->watering, 1=wet->don't watering
      if (!mqttresult) {
        DEBUG_PRINT("MQTT: state sensor 1 publish failed -> rc: ");
        DEBUG_PRINTLN(mqttresult);
      } else {
        DEBUG_PRINTLN("MQTT: state sensor 1 publish successful: ");
        DEBUG_PRINT(mqtt_wet_state_topic);DEBUG_PRINT(":");
        DEBUG_PRINTLN(String(!relays1_state).c_str());
      }
#else
        DEBUG_PRINTLN("MQTT: state sensor 1 publish disabled ...");
#endif
      mqtt_wet_state_topic[0] = '\0';
   
      // Publish wet state sensor 2
      strcpy (mqtt_wet_state_topic,mqtt_topic);
      strcat (mqtt_wet_state_topic,"/humid/master/state/2");
#ifdef MQTT_PUBLISH
      mqttresult = client.publish(mqtt_wet_state_topic, String(!relays2_state).c_str(), true);  // 0=dry->watering, 1=wet->don't watering
      if (!mqttresult) {
        DEBUG_PRINT("MQTT: state sensor 2 publish failed -> rc: ");
        DEBUG_PRINTLN(mqttresult);
      } else {
        DEBUG_PRINTLN("MQTT: state sensor 2 publish successful: ");
        DEBUG_PRINT(mqtt_wet_state_topic);DEBUG_PRINT(":");
        DEBUG_PRINTLN(String(!relays2_state).c_str());
      }
#else
        DEBUG_PRINTLN("MQTT: state sensor 2 publish disabled ...");
#endif
    }


    // Publish averageTresholdValue Group1
    if (sendTopic & MQTT_AVERAGETRESHOLD) {
      strcpy (mqtt_averageTresholdValueGroup_topic,mqtt_topic);
      strcat (mqtt_averageTresholdValueGroup_topic,"/humid/master/averageTresholdValueGroup/1");
#ifdef MQTT_PUBLISH
      mqttresult = client.publish(mqtt_averageTresholdValueGroup_topic, String(rtcMem.averageTresholdValueGroup1).c_str(), true);
      if (!mqttresult) {
        DEBUG_PRINT("MQTT: averageTresholdValue Group 1 publish failed -> rc: ");
        DEBUG_PRINTLN(mqttresult);
      } else {
        DEBUG_PRINTLN("MQTT: averageTresholdValue Group 1 publish successful: ");
        DEBUG_PRINT(mqtt_averageTresholdValueGroup_topic);DEBUG_PRINT(":");
        DEBUG_PRINTLN(String(rtcMem.averageTresholdValueGroup1).c_str());
      }
#else
        DEBUG_PRINTLN("MQTT: averageTresholdValue Group 1 publish disabled ...");
#endif

      mqtt_averageTresholdValueGroup_topic[0] = '\0';

      // Publish averageTresholdValue Group2"
      strcpy (mqtt_averageTresholdValueGroup_topic,mqtt_topic);
      strcat (mqtt_averageTresholdValueGroup_topic,"/humid/master/averageTresholdValueGroup/2");
#ifdef MQTT_PUBLISH
      mqttresult = client.publish(mqtt_averageTresholdValueGroup_topic, String(rtcMem.averageTresholdValueGroup2).c_str(), true);
      if (!mqttresult) {
        DEBUG_PRINT("MQTT: averageTresholdValue Group 2 publish failed -> rc: ");
        DEBUG_PRINTLN(mqttresult);
      } else {
        DEBUG_PRINTLN("MQTT: averageTresholdValue Group 2 publish successful: ");
        DEBUG_PRINT(mqtt_averageTresholdValueGroup_topic); DEBUG_PRINT(":");
        DEBUG_PRINTLN(String(rtcMem.averageTresholdValueGroup2).c_str());
      }
#else
        DEBUG_PRINTLN("MQTT: averageTresholdValue Group 2 publish disabled ...");
#endif
    }

    // Publish current version
    if (sendTopic & MQTT_VERSION) {
      strcpy (mqtt_version_topic,mqtt_topic);
      strcat (mqtt_version_topic,"/firmware/master/version");
#ifdef MQTT_PUBLISH
      mqttresult = client.publish(mqtt_version_topic, String(VERSION).c_str(), true);  
      if (!mqttresult) {
        DEBUG_PRINT("MQTT: current VERSION publish failed -> rc: ");
        DEBUG_PRINTLN(mqttresult);
      } else {
        DEBUG_PRINTLN("MQTT: current VERSION publish successful ...");
      }
#else
        DEBUG_PRINTLN("MQTT: current VERSION publish disabled ...");
#endif
    }

    // Set "I am awake" flag
    if (wakeup && !gosleep && (sendTopic & MQTT_IMAWAKE)) {
      strcpy (mqtt_iamawake_topic,mqtt_topic);
      strcat (mqtt_iamawake_topic,"/iamawake/master/00");
#ifdef MQTT_PUBLISH
      mqttresult = client.publish(mqtt_iamawake_topic, String(true).c_str(), true);  
      if (!mqttresult) {
         DEBUG_PRINT("MQTT: I am awake SET publish failed -> rc: ");
         DEBUG_PRINTLN(mqttresult);
      } else {
         DEBUG_PRINTLN("MQTT: I am awake SET publish successful ...");
      }
#else
         DEBUG_PRINTLN("MQTT: I am awake SET publish disabled ...");
#endif
    }

    // Reset wakeup flag
    if (wakeup && gosleep && (sendTopic & MQTT_WAKEUP)) {
      strcpy (mqtt_wakeup_topic,mqtt_topic);
      strcat (mqtt_wakeup_topic,"/wakeup/master/00");
#ifdef MQTT_PUBLISH
      mqttresult = client.publish(mqtt_wakeup_topic, String(!wakeup).c_str(), true);  
      if (!mqttresult) {
        DEBUG_PRINT("MQTT: wakeup RESET publish failed -> rc: ");
        DEBUG_PRINTLN(mqttresult);
      } else {
        DEBUG_PRINTLN("MQTT: wakeup RESET publish successful ...");
      }
#else
        DEBUG_PRINTLN("MQTT: wakeup RESET publish disabled ...");
#endif

      // Reset "I am awake" flag
      strcpy (mqtt_iamawake_topic,mqtt_topic);
      strcat (mqtt_iamawake_topic,"/iamawake/master/00");
#ifdef MQTT_PUBLISH
      mqttresult = client.publish(mqtt_iamawake_topic, String(false).c_str(), true);  
      if (!mqttresult) {
        DEBUG_PRINT("MQTT: I am awake RESET publish failed -> rc: ");
        DEBUG_PRINTLN(mqttresult);
      } else {
        DEBUG_PRINTLN("MQTT: I am awake RESET publish successful ...");
      }
#else
        DEBUG_PRINTLN("MQTT: I am awake RESET publish disabled ...");
#endif
    }
      
      // Send error
    if ((dont_switch_error1 || dont_switch_error2) && (sendTopic & MQTT_ERROR)) {
      strcpy (mqtt_error_topic,mqtt_topic);
      strcat (mqtt_error_topic,"/error/master/00");
#ifdef MQTT_PUBLISH
      mqttresult = client.publish(mqtt_error_topic, "Relay 1 or 2 has not switched. Check battery voltage!", true);  
      if (!mqttresult) {
        DEBUG_PRINT("MQTT: Error message publish failed -> rc: ");
        DEBUG_PRINTLN(mqttresult);
      } else {
        DEBUG_PRINTLN("MQTT: Relay 1 or 2 has not switched. Check battery voltage!");
      }
#else
        DEBUG_PRINTLN("MQTT: Error message publish disabled ...");
#endif
    }

    // Publish Last transmission Value
    // Transmit every time without flag
    // Publish time only if ntp in sync
    if (ntp_sync) {
        strcat (mqtt_last_transmission_topic,mqtt_topic);
        strcat (mqtt_last_transmission_topic,"/last-transmission/master/00");
#ifdef MQTT_PUBLISH
        mqttresult = client.publish(mqtt_last_transmission_topic, nowTime, true);
 
       if (!mqttresult) {
          DEBUG_PRINTLN("MQTT: mqtt_last_transmission_topic publish failed ...");
        } else {
         DEBUG_PRINTLN("MQTT: mqtt_last_transmission_topic publish successful ...");
        }
#else
        DEBUG_PRINTLN("MQTT: mqtt_last_transmission_topic publish disabled ...");
#endif
    } else {
        DEBUG_PRINTLN("MQTT mqtt_last_transmission_topic not publish - ntp not in sync.");
    }
    
    // Empty the array that strcat can work again
    mqtt_battery_topic[0] = '\0';                 
    mqtt_treshold_topic[0] = '\0';
    mqtt_wet_state_topic[0] = '\0';             
    mqtt_version_topic[0] = '\0';
    mqtt_last_transmission_topic[0] = '\0';
    mqtt_wakeup_topic[0] = '\0';
    mqtt_iamawake_topic[0] = '\0';
    mqtt_error_topic[0] = '\0'; 
    mqtt_averageTresholdValueGroup_topic[0] = '\0';

    DEBUG_PRINT("MQTT: Leave sendMQTT(");
    DEBUG_PRINT(sendTopic);
    DEBUG_PRINTLN(")");
  }
}


// **************************************************************************
//  MQTT subscribing
// **************************************************************************

void subscribeMQTT() {
    
  char sensornr[2] = "";
  boolean rc = false;

  char mqtt_moisture_topic[40] = "";
  char mqtt_battery_topic[40] = "";
  char mqtt_wakeup_topic[40] = "";
  char mqtt_humidCalc1_topic[40] = "";
  char mqtt_humidCalc2_topic[40] = "";
  char mqtt_lastTransmission_topic[40] = "";
  
  DEBUG_PRINTLN("Subcribing topics....");
  // Subscribing wakeup for master
  strcpy (mqtt_wakeup_topic,mqtt_topic);
  strcat (mqtt_wakeup_topic,"/wakeup/master/00");          // Own topic for wakeup
  
  // Specify the QoS to subscribe at. Only supports QoS 0 or 1:
  DEBUG_PRINTLN(mqtt_wakeup_topic);
   
  rc = client.subscribe(mqtt_wakeup_topic,1);            // Subscribe wakeup flag

  if (!rc) {
    DEBUG_PRINTLN("MQTT: wakeup state subscribe failed ...");
  } else {
    DEBUG_PRINTLN("MQTT: wakeup state subscribe successful ...");
  }

  // Subscribing moisture of all Sensors
  strcpy (mqtt_moisture_topic,mqtt_topic);
  strcat (mqtt_moisture_topic,"/moisture/sensor/#");    // All topics of sensor
  
  // Specify the QoS to subscribe at. Only supports QoS 0 or 1:
  DEBUG_PRINTLN(mqtt_moisture_topic);

  rc = client.subscribe(mqtt_moisture_topic,1);         // Subscrive all frequency values

  if (!rc) {
    DEBUG_PRINTLN("MQTT: moisture subscribe failed ...");
  } else {
    DEBUG_PRINTLN("MQTT: moisture subscribe successful ...");
  }

  // Subscribing battery state of all Sensors
  strcpy (mqtt_battery_topic,mqtt_topic);
  strcat (mqtt_battery_topic,"/battery/sensor/#");          // All topics of battery
  
  // Specify the QoS to subscribe at. Only supports QoS 0 or 1:
  DEBUG_PRINTLN(mqtt_battery_topic);
   
  rc = client.subscribe(mqtt_battery_topic,1);             // Subscribe all Battery values

  if (!rc) {
    DEBUG_PRINTLN("MQTT: battery state subscribe failed ...");
  } else {
    DEBUG_PRINTLN("MQTT: battery state subscribe successful ...");
  }

  // Subscribing enaHumidCalc1 for Master
  strcpy (mqtt_humidCalc1_topic,mqtt_topic);
  strcat (mqtt_humidCalc1_topic,"/humid/master/enaHumidCalcSensor1/00");          // Topic for enable/disable humid calculation for Relay 1
  
  // Specify the QoS to subscribe at. Only supports QoS 0 or 1:
  DEBUG_PRINTLN(mqtt_humidCalc1_topic);
   
  rc = client.subscribe(mqtt_humidCalc1_topic,1);            // Subscribe enaHumidCalc1 flag

  if (!rc) {
    DEBUG_PRINTLN("MQTT: enaHumidCalc1 state subscribe failed ...");
  } else {
    DEBUG_PRINTLN("MQTT: enaHumidCalc1 state subscribe successful ...");
  }


  // Subscribing enaHumidCalc2 for Master
  strcpy (mqtt_humidCalc2_topic,mqtt_topic);
  strcat (mqtt_humidCalc2_topic,"/humid/master/enaHumidCalcSensor2/00");          // Topic for enable/disable humid calculation for Relay 2
  
  // Specify the QoS to subscribe at. Only supports QoS 0 or 1:
  DEBUG_PRINTLN(mqtt_humidCalc2_topic);
   
  rc = client.subscribe(mqtt_humidCalc2_topic,1);            // Subscribe enaHumidCalc1 flag

  if (!rc) {
    DEBUG_PRINTLN("MQTT: enaHumidCalc2 state subscribe failed ...");
  } else {
    DEBUG_PRINTLN("MQTT: enaHumidCalc2 state subscribe successful ...");
  }

  // Subscribing moisture of all Sensors
  strcpy (mqtt_moisture_topic,mqtt_topic);
  strcat (mqtt_moisture_topic,"/last-transmission/sensor/#");    // All topics of sensor
  
  // Specify the QoS to subscribe at. Only supports QoS 0 or 1:
  DEBUG_PRINTLN(mqtt_moisture_topic);

  rc = client.subscribe(mqtt_moisture_topic,1);         // Subscrive all frequency values

  if (!rc) {
    DEBUG_PRINTLN("MQTT: last-transmission for all sensors subscribe failed ...");
  } else {
    DEBUG_PRINTLN("MQTT: last-transmission for all sensors subscribe successful ...");
  }

  
  // Empty the arrays that strcat can work again            
  mqtt_moisture_topic[0] = '\0';                 
  mqtt_battery_topic[0] = '\0';                  
  mqtt_wakeup_topic[0] = '\0';                   
  mqtt_humidCalc1_topic[0] = '\0';               
  mqtt_humidCalc2_topic[0] = '\0';               
  mqtt_lastTransmission_topic[0] = '\0';
  client.loop();
  delay(10);                        // Wait for receive messages
}


// **************************************************************************
//  MQTT reconnect
// **************************************************************************
void reconnectMQTT() {
  int i = 0;
  if (mqtt_enabled) {
    while (!client.connected())
    {
      DEBUG_PRINT("Attempting MQTT connection...");
      if (client.connect(host_name))
      { 
        DEBUG_PRINTLN("connected");
        subscribeMQTT();
      }
      else
      {
        DEBUG_PRINT("failed, rc=");
        DEBUG_PRINTLN(client.state());
        DEBUG_PRINTLN(" try again in 5 seconds");
        delay(5000);
        if ( i >= 10 ) {
          break;
          DEBUG_PRINTLN("MQTT-Connect timeout");
        }
        DEBUG_PRINT(++i);
        DEBUG_PRINT(" ");
      }
    }
  } else {
     DEBUG_PRINTLN("MQTT disabled!");
  }
}


// **************************************************************************
//  Get Status of Relay 0 Ohm = wet, 10 kOhm = dry
// **************************************************************************
//#define STATUS_1_IN_0           D5    // Low for Status 0 Ohm for Sensor 1 -> wet (GPIO14)    -> don't watering
                                        // High for Status 10 kOhm for Sensor 1 -> dry (GPIO14) -> watering
//#define STATUS_2_IN_0           D7    // Low for Status 0 Ohm for Sensor 2 -> wet (GPIO13)    -> don't watering                                                                                                                                                                                                                              // High for Status 0 Ohm for Sensor 2 -> dry (GPO13)
                                        // High for Status 10 kOhm for Sensor 2 -> dry (GPIO13) -> watering

void getRelayState() {
  DEBUG_PRINTLN("Get state of bistabil relays ...");

  // negative Logik - Pullups  
  DEBUG_PRINT("STATUS_1_IN_0 = ");
  DEBUG_PRINTLN(!digitalRead(STATUS_1_IN_0));
  DEBUG_PRINT("STATUS_2_IN_0 = ");
  DEBUG_PRINTLN(!digitalRead(STATUS_2_IN_0));

  // State sensor 1
  if (digitalRead(STATUS_1_IN_0)) {
    relays1_state = false;                          // dry -> watering
    DEBUG_PRINTLN("relays1_state = false -> dry -> watering");
  }
  else {
    relays1_state = true;                          // wet -> don't watering
    DEBUG_PRINTLN("relays1_state = true -> wet -> don't watering");
  }

  //State sensor 2
  if (digitalRead(STATUS_2_IN_0)) {
    relays2_state = false;                          // dry -> watering
    DEBUG_PRINTLN("relays2_state = false -> dry -> watering");
  }
  else {
    relays2_state = true;                          // wet -> don't watering
    DEBUG_PRINTLN("relays2_state = true -> wet -> don't watering");
  }
  sendMQTT(MQTT_STATE);                            //send stats via MQTT
}


// **************************************************************************
//  Output Moisture to GARDENA MultiControl 0 Ohm = Dry, 10 kOhm = wet
// ******************Batterie********************************************************
void outputMoisture() {

  int i;
  float sumTresholdValueGroup1 = 0;
  float sumTresholdValueGroup2 = 0;
  int countAvailableSensorGroup1 = 0;   
  int countAvailableSensorGroup2 = 0;   
  int countReceivedSensorsGroup1 = 0;
  int countReceivedSensorsGroup2 = 0;

  char nowTime[6];
  sprintf(nowTime, "%02d:%02d", hour(now()), minute(now()));

  DEBUG_PRINTLN("OUTPUT_MOISTURE: enter");
  DEBUG_PRINTLN("Calculate wet/dry for output to bistabil relays 1&2...");

  // Check if the interval overflow into tomorrow
  if ( Time2Minutes(blockingtime_to) > Time2Minutes(blockingtime_from) )
  {
    disableTresholdCalculation = (Time2Minutes(nowTime) >= Time2Minutes(blockingtime_from) && Time2Minutes(nowTime) < Time2Minutes(blockingtime_to));
  }
  else
  {
    disableTresholdCalculation = (Time2Minutes(nowTime) >= Time2Minutes(blockingtime_from) || Time2Minutes(nowTime) < Time2Minutes(blockingtime_to));
  }

  //calculate average sensevalue
  for (i = 1; i <= sensors_count ; i++) {
    DEBUG_PRINT("Sensor ");
    DEBUG_PRINT(i);
    DEBUG_PRINT(" | avalaible: ");
    DEBUG_PRINT(sensorAvailable[i]);
    DEBUG_PRINT(" | frequency: ");
    DEBUG_PRINT(freqValue[i]);
    DEBUG_PRINT(" | treshold: ");
    DEBUG_PRINT(sensorTreshold[i]);
    DEBUG_PRINT(" | weighting: ");
    DEBUG_PRINT(sensorWeighting[i]);
    DEBUG_PRINT(" | wet/dry: ");
    DEBUG_PRINT((freqValue[i] < sensorTreshold[i])? "wet":"dry");
    DEBUG_PRINT(" | group: ");
    DEBUG_PRINTLN(sensorGroup[i]);
    
    // Count available sensors and calculate avarage Treshold for sensor group 1 and group 2
    // 1 means wet 
    // 0 means dry
    // result [% wet] 0=dry....1=wet
    if (sensorAvailable[i] && (freqValue[i] > 0) && (sensorGroup[i] == 1)){
       sumTresholdValueGroup1 += sensorWeighting[i]/100 * (freqValue[i] < sensorTreshold[i])? 1:0;
       countAvailableSensorGroup1++;
       receivedFreqValue[i] = true;                   
    }

    if (sensorAvailable[i] && (freqValue[i] > 0) && (sensorGroup[i] == 2)){  
       sumTresholdValueGroup2 += sensorWeighting[i]/100 * (freqValue[i] < sensorTreshold[i])? 1:0;
       countAvailableSensorGroup2++;
       receivedFreqValue[i] = true;                   
    }
  } // end for (...sensor_count...)

  // Deactivate new calculation of sprinkler control as long as sprinkler programme is running
  if (!disableTresholdCalculation) { 
    // prevent div by zero and calculate new value and overwrite loaded value from rtcMem
    rtcMem.averageTresholdValueGroup1 = (countAvailableSensorGroup1 == 0) ? -1 :  sumTresholdValueGroup1/countAvailableSensorGroup1;
    writeToRTCMemory(); // save calculated averageTresholdValueGroup1/averageTresholdValueGroup2 to use if calculation is disabled
#ifdef DEBUG         
  } else {
    DEBUG_PRINTLN("Average treshold calculation deactivated");
    DEBUG_PRINT("Use average treshold from rtcMem group 1: ");
    DEBUG_PRINTLN(rtcMem.averageTresholdValueGroup1);
#endif
  }
  
  DEBUG_PRINTLN("--------------------------");
  DEBUG_PRINT("sumTresholdValueGroup1: ");
  DEBUG_PRINTLN(sumTresholdValueGroup1);     
  DEBUG_PRINT("Humid Calc Senor 1 enabled: ");
  DEBUG_PRINTLN(enableHumidCalculationSensor1);
  DEBUG_PRINT("Availabel sensors Group 1: ");
  DEBUG_PRINTLN(countAvailableSensorGroup1);
  DEBUG_PRINT("Average sensevalue Group 1: ");
  DEBUG_PRINTLN(rtcMem.averageTresholdValueGroup1);
  DEBUG_PRINTLN("--------------------------");

  // Deactivate new calculation of sprinkler control as long as sprinkler programme is running
  if (!disableTresholdCalculation) { 
    // prevent div by zero and calculate new value and overwrite loaded value from rtcMem
    rtcMem.averageTresholdValueGroup2 = (countAvailableSensorGroup2 == 0) ? -1 : sumTresholdValueGroup2/countAvailableSensorGroup2;
    writeToRTCMemory(); // save calculated averageTresholdValueGroup1/averageTresholdValueGroup2 to use if calculation is disabled
#ifdef DEBUG         
  } else {
    DEBUG_PRINTLN("Average treshold calculation deactivated");
    DEBUG_PRINT("Use average treshold from rtcMem group 2: ");
    DEBUG_PRINTLN(rtcMem.averageTresholdValueGroup2);
#endif
  }
  
  DEBUG_PRINT("sumTresholdValueGroup2: ");
  DEBUG_PRINTLN(sumTresholdValueGroup2);     
  DEBUG_PRINT("Humid Calc Senor 2 enabled: ");
  DEBUG_PRINTLN(enableHumidCalculationSensor2);
  DEBUG_PRINT("Availabel sensors Group 2: ");
  DEBUG_PRINTLN(countAvailableSensorGroup2);
  DEBUG_PRINT("Average sensevalue Group 2: ");
  DEBUG_PRINTLN(rtcMem.averageTresholdValueGroup2);
  DEBUG_PRINTLN("--------------------------");

  // Sensor group 1 
  // Topic /PSMSense/humid/master/enaHumidCalcSensor1/00 = 1 (true)
  // watering for averageTresholdValueGroup1 < 1
  DEBUG_PRINTLN("OUTPUT_MOISTURE: Set/Reset bistable relays 1/2");
  if (enableHumidCalculationSensor1)  {
      DEBUG_PRINTLN("-------------START RELAYS 1---------");
      DEBUG_PRINTLN("enableHumidCalculationSensor1");
      DEBUG_PRINT("Treshold sensor group 1: ");
      DEBUG_PRINTLN(tresholdGroup1);
      // Moisture gt n kHz (averageTresholdValueGroup1) = dry; relays1_state = true (V1=Off)
      // Moisture lt n kHz (averageTresholdValueGroup1) = wet; relays1_state = false (V1=On)
      if (countAvailableSensorGroup1 > 0) {
        if ((rtcMem.averageTresholdValueGroup1 < tresholdGroup1) && (relays1_state)) {
          digitalWrite(RESET_RELAYS1, LOW);               // reset relay 1 -> 10 kOhm -> dry
          DEBUG_PRINT("reset relays 1 to off (dry)-->");
          DEBUG_PRINTLN(relays1_state);
          if (!relays1_state) {                           // check if relais has switch off, if no set error for sendMQTT()
            dont_switch_error1=true;
          } else {  
            dont_switch_error1=false;                     // reset error
          }
        } else if ((rtcMem.averageTresholdValueGroup1 >= tresholdGroup1) && (relays1_state==false)){
          digitalWrite(SET_RELAYS1, LOW);                 // set relay 1 ->  0 Ohm -> wet
          DEBUG_PRINT("set relays 1 to on (wet)-->");
          DEBUG_PRINTLN(relays1_state);
          if (relays1_state) {                            // check if relais has switch on, if no set error for sendMQTT()
            dont_switch_error1=true;
          } else {  
            dont_switch_error1=false;                     // reset error
          }
        } else {
          DEBUG_PRINT("relays1_state: ");
          DEBUG_PRINTLN(relays1_state);
          DEBUG_PRINT("averageTresholdValueGroup1: ");
          DEBUG_PRINTLN(rtcMem.averageTresholdValueGroup1);
        }  
        delay(500);
        digitalWrite(SET_RELAYS1, HIGH);                   // GPIO02 (set)   relay 1 off 
        digitalWrite(RESET_RELAYS1, HIGH);                 // GPIO05 (reset) relay 1 off 
      }
  } else {
      // Topic /PSMSense/humid/master/enaHumidCalcSensor1/00 = 0 (false)
      // Only output switching pulse if status not yet reached (save power)
      DEBUG_PRINTLN("not enableHumidCalculationSensor1");
      if (relays1_state==false) {
         digitalWrite(SET_RELAYS1, LOW);                  // set relay 1 -> 0 Ohm -> wet
         delay(500);
         digitalWrite(SET_RELAYS1, HIGH);                 // GPIO02 (set) relay 1 off
         DEBUG_PRINTLN("set relays 1 to on (wet)");
#ifdef DEBUG         
      } else {
         DEBUG_PRINTLN("relays 1 is off: don't switch. ");  
         DEBUG_PRINT("State: "); 
         DEBUG_PRINTLN(relays1_state);
#endif          
      }   
  }
  DEBUG_PRINTLN("-------------END RELAYS 1-----------");

  
  // Sensor group 2 
  // Topic /PSMSense/humid/master/enaHumidCalcSensor2/00 = 1 (true)
  // watering for averageTresholdValueGroup2 < 1
  if (enableHumidCalculationSensor2)  {
      DEBUG_PRINTLN("-------------START RELAYS 2---------");
      DEBUG_PRINTLN("enableHumidCalculationSensor2");
      DEBUG_PRINT("Treshold sensor group 2: ");
      DEBUG_PRINTLN(tresholdGroup2);
      // Moisture gt n kHz (averageTresholdValueGroup2) = dry; relays2_state = true (V2=Off)
      // Moisture lt n kHz (averageTresholdValueGroup2) = wet; relays2_state = false (V2=On)
      if (countAvailableSensorGroup2 > 0) {
        if ((rtcMem.averageTresholdValueGroup2 < tresholdGroup2) && (relays2_state)) {
          digitalWrite(RESET_RELAYS2, LOW);               // reset relay 2 -> 10 kOhm -> dry
          DEBUG_PRINT("reset relays 2 to off (dry)-->");
          DEBUG_PRINTLN(relays2_state);
          if (!relays2_state) {                           // check if relais has switch off, if no set error for sendMQTT()
            dont_switch_error2=true;
          } else {  
            dont_switch_error2=false;                    // reset error
          }
        } else if ((rtcMem.averageTresholdValueGroup2 >= tresholdGroup2) && (relays2_state==false)){
          digitalWrite(SET_RELAYS2, LOW);                // set relay 2 -> 0 Ohm -> wet 
          DEBUG_PRINT("set relays 2 to on (wet)-->");
          DEBUG_PRINTLN(relays2_state);
          if (relays2_state) {                           // check if relais has switch on, if no set error for sendMQTT()
            dont_switch_error2=true;
          } else {  
            dont_switch_error2=false;                    // reset error
          }
        } else {
          DEBUG_PRINT("relays2_state: ");
          DEBUG_PRINTLN(relays2_state);
          DEBUG_PRINT("averageTresholdValueGroup2: ");
          DEBUG_PRINTLN(rtcMem.averageTresholdValueGroup2);
        }  
        delay(500);
        digitalWrite(SET_RELAYS2, HIGH);                   // GPIO00 (set)   relay 2 off
        digitalWrite(RESET_RELAYS2, HIGH);                 // GPIO04 (reset) relay 2 off
      }
  } else {
      // Topic /PSMSense/humid/master/enaHumidCalcSensor2/00 = 0 (false)
      // Only output switching pulse if status not yet reached (save power)
      DEBUG_PRINTLN("not enableHumidCalculationSensor2");
      if (relays2_state==false) {
         digitalWrite(SET_RELAYS2, LOW);                  // set relay 2 -> 0 Ohm -> wet
         delay(500);
         digitalWrite(SET_RELAYS2, HIGH);                 // GPIO00 (set) relay off
         DEBUG_PRINTLN("set relays 2 to on (wet)");
#ifdef DEBUG         
      } else {
         DEBUG_PRINTLN("relays 2 is off: don't switch. ");  
         DEBUG_PRINT("State: "); 
         DEBUG_PRINTLN(relays2_state);
#endif          
      }   
   }
   DEBUG_PRINTLN("-------------END RELAYS 2-----------");
   getRelayState();

  // check if all sensor recieved
  // then send treshold for group 1 & 2
  // -> reduce different values in home atomation datastore
  if (mqtt_enabled){
    for (i = 1; i <= sensors_count ; i++) {
      if ((sensorGroup[i] == 1) && receivedFreqValue[i]) {
         countReceivedSensorsGroup1++;
      } else if ((sensorGroup[i] == 2) && receivedFreqValue[i]) {
         countReceivedSensorsGroup2++;
      } else {
         DEBUG_PRINTLN("OUTPUT_MOISTURE: Error count received sensor data");
         DEBUG_PRINT("OUTPUT_MOISTURE: sensorGroup[");DEBUG_PRINT(i);DEBUG_PRINT("]: ");DEBUG_PRINTLN(sensorGroup[i]);
         DEBUG_PRINT("OUTPUT_MOISTURE: receivedFreqValue[");DEBUG_PRINT(i);DEBUG_PRINT("]: ");DEBUG_PRINTLN(receivedFreqValue[i]);
      }
    }
  } else {
    DEBUG_PRINTLN("MQTT disabled!");
  }
  DEBUG_PRINT("OUTPUT_MOISTURE: count received sensors group 2: ");
  DEBUG_PRINTLN(countReceivedSensorsGroup2);
  DEBUG_PRINT("OUTPUT_MOISTURE: count received sensors group 1: ");
  DEBUG_PRINTLN(countReceivedSensorsGroup1);

  // if all sensor values are recieved send treshold results to MQTT broker
  if ((countReceivedSensorsGroup1 + countReceivedSensorsGroup2) == sensors_count) {
     DEBUG_PRINTLN("OUTPUT_MOISTURE: Publish MQTT_AVERAGETRESHOLD (4)");
     sendMQTT(MQTT_AVERAGETRESHOLD); 
  }
   
  if (dont_switch_error1 || dont_switch_error2){
     sendMQTT(MQTT_ERROR);    
  }
  //reconnectMQTT(); 
  DEBUG_PRINTLN("OUTPUT_MOISTURE: exit");
}

// **************************************************************************
//  Self-OTA-Update
// **************************************************************************
void strremove(char* source,char ch) { 
        char* target=source;
        for (;(*target=*source)!=0;source++)
                if (*target!=ch) target++;
}

void checkForUpdates() {
  uint8_t mac[6];
  char macAddr[14];
  char chVERSION[6];
  char fwUrlBase[50];
  int FW_VERSION;  

  //Remove dots from const string VERSION
  strcpy(chVERSION, VERSION.c_str()); 
  strremove(chVERSION,'.');
  FW_VERSION =atoi(chVERSION);
    
  WiFi.macAddress( mac );
  sprintf(macAddr, "%02X%02X%02X%02X%02X%02X", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]); /// capital letters at MAC address
  strcpy(fwUrlBase, "http://");
  strcat(fwUrlBase, otau_server);
  strcat(fwUrlBase, otau_path);
  String (fwURL) = String(fwUrlBase);
  fwURL.concat( macAddr );
  String fwVersionURL = fwURL;
  fwVersionURL.concat( ".version" );
  
  DEBUG_PRINTLN( "Checking for firmware updates." );

  DEBUG_PRINT( "MAC address: " );
  DEBUG_PRINTLN( macAddr );
  DEBUG_PRINT( "Firmware version URL: " );
  DEBUG_PRINTLN( fwVersionURL );

  HTTPClient httpClient;
  httpClient.begin(espClient, fwVersionURL );
  int httpCode = httpClient.GET();
  if( httpCode == 200 ) {
    String newFWVersion = httpClient.getString();

    DEBUG_PRINT( "Current firmware version: " );
    DEBUG_PRINTLN( FW_VERSION );
    DEBUG_PRINT( "Available firmware version: " );
    DEBUG_PRINTLN( newFWVersion );

    int newVersion = newFWVersion.toInt();

    if( newVersion > FW_VERSION ) {
      DEBUG_PRINTLN( "Preparing to update" );

      String fwImageURL = fwURL;
      fwImageURL.concat( ".bin" );
      t_httpUpdate_return ret = ESPhttpUpdate.update(espClient, fwImageURL );

#ifdef DEBUG
      switch(ret) {
        case HTTP_UPDATE_FAILED:
          Serial.printf("HTTP_UPDATE_FAILD Error (%d): %s", ESPhttpUpdate.getLastError(), ESPhttpUpdate.getLastErrorString().c_str());
          DEBUG_PRINTLN( "Update canceled!" );
          break;

        case HTTP_UPDATE_NO_UPDATES:
          Serial.println("HTTP_UPDATE_NO_UPDATES");
          DEBUG_PRINTLN( "Update canceled!" );
          break;
      }
#endif      
    }
    else {
      DEBUG_PRINTLN( "Already on latest version" );
    }
  }
  else {
    DEBUG_PRINT( "Firmware version check failed, got HTTP response code " );
    DEBUG_PRINTLN( httpCode );
  }
  httpClient.end();
}

// **************************************************************************
//  Setup: battery measuring
// **************************************************************************
// genaue Spannung der Stromquelle, PIN A0 muss mit 5 V über ca. 180kOhm Verbunden werden! 
// Summe R=5kOhm - Spannungsteiler auf Board 100k/220k +  extern 180k
void measureBatt() { 
  
  // Read battery voltage (5000 = 5000mV)
  DEBUG_PRINTLN("Starting Battery measurement ...");
  
  DEBUG_PRINT("Battery voltage RAW: ");
  DEBUG_PRINTLN(analogRead(A0));

  analogValue[0] = (float)(map(analogRead(A0),0,1023,0,5000))/1000+adc_shift;

  DEBUG_PRINT("Battery voltage correction: ");
  DEBUG_PRINTLN(adc_shift);

  DEBUG_PRINT("Battery voltage: ");
  DEBUG_PRINTLN(analogValue[0]);
  sendMQTT(MQTT_BATTERY); 
}

// **************************************************************************
//  Setup: Enable the Web Services, only in Power Supply Mode or when wakeup over MQTT
// **************************************************************************
void enableWebService() {  
  if (powermode || wakeup) { 
      DEBUG_PRINTLN("Enable WebService ...");
      // Enable the Free Memory Page
      httpServer.on("/freemem", []() {
      DEBUG_PRINTLN("WEB: Connection received: /freemem : ");
      DEBUG_PRINT(ESP.getFreeSketchSpace());
      httpServer.sendHeader("Connection", "close");
      httpServer.send(200, "text/plain", String(ESP.getFreeSketchSpace()).c_str());
    });

    httpServer.on("/reboot", Handle_Reboot);              // Reboot the ESP Controller
    httpServer.on("/gosleep", Handle_GoSleep);            // Reset Wakeup Topic to GoSleep after config ESP etc.
    httpServer.on("/clearconfig", Handle_ClearConfig);    // Delete the Json config file on Filesystem
    httpServer.on("/config", Handle_config);              // Show the configuration page
    httpServer.on("/getvalue", Handle_getvalue);          // Return the current frequency value as plain text without html header
    httpServer.on("/", Handle_welcome);                   // Show the main page
    httpServer.on("/sensor_config", Handle_sensor_config); // Show the sensor_config page
    httpServer.on("/sysconf", Handle_SysConf);            // Show system config
    httpUpdater.setup(&httpServer);                       // Enable the OTA
    httpServer.begin();                                   // Enable the WebServer
  }

}

// *************************************************************************
// Read/Write from/to RTCMemory
// *************************************************************************
void readFromRTCMemory() {
  system_rtc_mem_read(RTCMEMORYSTART, &rtcMem, sizeof(rtcMem));

  DEBUG_PRINT("Read averageTresholdValueGroup1 = ");
  DEBUG_PRINTLN(rtcMem.averageTresholdValueGroup1);
  DEBUG_PRINT("Read averageTresholdValueGroup2 = ");
  DEBUG_PRINTLN(rtcMem.averageTresholdValueGroup2);
  yield();
}

void writeToRTCMemory() {
  system_rtc_mem_write(RTCMEMORYSTART, &rtcMem, sizeof(rtcMem));

  DEBUG_PRINT("Wrote averageTresholdValueGroup1 = ");
  DEBUG_PRINTLN(rtcMem.averageTresholdValueGroup1);
  DEBUG_PRINT("Wrote averageTresholdValueGroup2 = ");
  DEBUG_PRINTLN(rtcMem.averageTresholdValueGroup2);
  yield();
}

// **************************************************************************
//  Set the measure flag, called by ticker
// **************************************************************************
void set_measure_flag() {
  func_measure_call = true;
}

// **************************************************************************
//  Setup
// **************************************************************************
void setup(void) {
  // put your setup code here, to run once:

  #ifdef DEBUG
//    Serial.begin(serialspeed, SERIAL_8N1, SERIAL_TX_ONLY, TX);
    Serial.begin(serialspeed);
    Serial.println();
    Serial.println("Booting PSMSense-Receiver V " + VERSION + " ...");
  #endif

  pinMode(SET_RELAYS1, OUTPUT);
  pinMode(RESET_RELAYS1, OUTPUT);
  pinMode(SET_RELAYS2, OUTPUT);
  pinMode(STATUS_1_IN_0, INPUT_PULLUP);
  pinMode(STATUS_2_IN_0, INPUT_PULLUP);
  pinMode(CLEAR_BTN, INPUT_PULLUP);
  pinMode(BATT_IN_ANALOG, INPUT);
  pinMode(POWER_MODE, INPUT_PULLUP);
  pinMode(WAKEUP, WAKEUP_PULLUP);
   
  digitalWrite(SET_RELAYS1, HIGH);                   // Set Relay 1 off
  digitalWrite(RESET_RELAYS1, HIGH);                 // Reset Relay 1 off
  digitalWrite(SET_RELAYS2, HIGH);                   // Set Relay 2 off
  
  if (digitalRead(POWER_MODE) == HIGH) {
    powermode = true;
  } else {
    powermode = false;
  }

  for (int i = 0; i < 10; i++) {
    receivedFreqValue[i] = false;                         // initialize array for received data
  }

  WiFiManager wifiManager;
  if (digitalRead(CLEAR_BTN) == LOW) wifiManager.resetSettings();   // Clear WIFI data if CLEAR Button is pressed during boot
  
  pinMode(RESET_RELAYS2, OUTPUT);                         // after check for pressing clear button change GPIO4 to output for RESET relay 2
  digitalWrite(RESET_RELAYS2, HIGH);                      // Reset Relay 2 off

  wifiManager.setAPCallback(configModeCallback);          // set callback that gets called when connecting to previous WiFi fails, and enters Access Point mode
  wifiManager.setSaveConfigCallback(saveConfigCallback);  // set config save notify callback
  
  if (LittleFS.begin()) {                                   // Mounting File System
    DEBUG_PRINTLN("SETUP: mounted file system");
    DEBUG_PRINTLN("SETUP: Load config....");
    if (!loadConfig()) {
      DEBUG_PRINTLN("SETUP: Failed to load config");
    } else {
      DEBUG_PRINTLN("SETUP: Config loaded");
    }

    // read rtcMem.averageTresholdValueGroup1 and rtcMem.averageTresholdValueGroup2

    DEBUG_PRINTLN("SETUP: Load system_config from rtcMem....");
    readFromRTCMemory();
       
    DEBUG_PRINTLN("SETUP: Load Master-Config");
    if (!loadSensorConfig()) {
      DEBUG_PRINTLN("SETUP: Failed to load Master-Config");
    } else {
      DEBUG_PRINTLN("SETUP: Master-Config loaded");
    }

  }
  else {
    DEBUG_PRINTLN("SETUP: failed to mount FS");
  }

  // Configure some additional 
  WiFiManagerParameter custom_hostname("hostname", "Optional Hostname", host_name, 20);
  wifiManager.addParameter(&custom_hostname);
  WiFiManagerParameter custom_mqttserver("mqttserver", "Optional MQTT Server", mqtt_server, 30);
  wifiManager.addParameter(&custom_mqttserver);
  WiFiManagerParameter custom_mqtttopic("mqtttopic", "Optional MQTT Topic", mqtt_topic, 30);
  wifiManager.addParameter(&custom_mqtttopic);
  WiFiManagerParameter custom_ntpserver("ntpserver", "Optional NTP Server", ntpserver, 30);
  wifiManager.addParameter(&custom_ntpserver);
  WiFiManagerParameter custom_otauserver("otuaserver", "Optional OTA-Update Server", otau_server, 30);
  wifiManager.addParameter(&custom_otauserver);

  String autoconf_ssid = "PSMMaster_Config_" + String(ESP.getChipId());
  wifiManager.setConnectTimeout(60);                                  // Workaround Test for reconnect issue
  wifiManager.setConfigPortalTimeout(120);                            // Timeout for SoftAP, try connect again to stored wlan

  //automatically connect using saved credentials if they exist
  //If connection fails it starts an access point with the specified name
  //here  "AutoConnectAP" if empty will auto generate basedcon chipid, if password is blank it will be anonymous
  //and goes into a blocking loop awaiting configuration
  int i = 5;                                                          // wait max. i * 3000 ms for w-lan connection
  if (!wifiManager.autoConnect(autoconf_ssid.c_str())) {               // Use PSMMaster_Config_+Chip ID as AP-name with 192.168.4.1 as IP - otherwise saved credetials
    delay(3000);
    if (i =0){                                                            // if connection failed go to sleep
        DEBUG_PRINTLN("failed to connect and hit timeout");
        ESP.deepSleep(60000000*bat_sampling_interval);                    // 60s * Value in Minutes from config
        yield();
        delay(100);                                                       // Only to ensure the Deepsleep is working
    }
    i--;    
  }
                       
  
  // Read and save the new values from AP config page
  strncpy(host_name, custom_hostname.getValue(), 20);
  strncpy(mqtt_server, custom_mqttserver.getValue(), 30);
  strncpy(mqtt_topic, custom_mqtttopic.getValue(), 30);
  strncpy(ntpserver, custom_ntpserver.getValue(), 30);
  strncpy(otau_server, custom_otauserver.getValue(), 30);

  DEBUG_PRINTLN("WiFi connected! IP: " + ipToString(WiFi.localIP()));
  DEBUG_PRINTLN("Hostname: " + String(host_name));
  DEBUG_PRINTLN("NTP-Server: " + String(ntpserver));
  DEBUG_PRINTLN("MQTT Server: " + String(mqtt_server));
  DEBUG_PRINTLN("MQTT Topic: " + String(mqtt_topic));
  if (mqtt_enabled) {
    DEBUG_PRINTLN("MQTT enabled");
  } else {
    DEBUG_PRINTLN("MQTT disabled");
  }
  DEBUG_PRINTLN("OTAU-Server: " + String(otau_server));

  // save the custom parameters to FS
  if (shouldSaveConfig) {
    DEBUG_PRINTLN(" config...");
    DynamicJsonBuffer jsonBuffer;
    JsonObject& json = jsonBuffer.createObject();
    json["hostname"] = host_name;
    json["mqttserver"] = mqtt_server;
    json["mqtttopic"] = mqtt_topic;
    json["ntpserver"] = ntpserver;
    json["otauserver"] = otau_server;

    File configFile = LittleFS.open("/config.json", "w");
    if (!configFile) {
      DEBUG_PRINTLN("SPI: failed to open config file for writing config");
    }
    #ifdef DEBUG
      json.printTo(Serial);
    #endif
    DEBUG_PRINTLN("");
    json.printTo(configFile);
    configFile.close();
  }

  if (host_name[0] == 0 ) strncpy(host_name, "PSMSense", 20);         //set default hostname when not set!
  if (mqtt_topic[0] == 0 ) strncpy(mqtt_topic, defaultmqtttopic, 30); //set defaultmqtttopic when not set!
  if (ntpserver[0] == 0 ) strncpy(ntpserver, poolServerName, 30);     //set default ntp server when not set!

  enableWebService();

// **************************************************************************
//  Setup: MQTT
// **************************************************************************

  
  if ((mqtt_enabled) && (strlen(mqtt_server) > 2)){       // > 2 means not empty ;-)
    client.setServer(mqtt_server, 1883);                  // Set the MQTT Broker
    client.setCallback(callbackMQTT);
    while (!client.connected()) {                // Wait for MQTT-Connetion
      reconnectMQTT();
    }
  }

// **************************************************************************
//  Setup: NTP
// **************************************************************************


  if (ntp_enabled) {
      DEBUG_PRINTLN("NTP: Starting UDP");
      Udp.begin(NtpLocalPort);
      DEBUG_PRINT("NTP: Local port: ");
      DEBUG_PRINTLN(Udp.localPort());
      DEBUG_PRINTLN("NTP: waiting for sync");
      setSyncProvider(getNtpTime);                       // set the external time provider
      setSyncInterval(3600);                             // set the number of seconds between re-sync
      //String boottimetemp = printDigits2(hour()) + ":" + printDigits2(minute()) + " " + printDigits2(day()) + "." + printDigits2(month()) + "." + String(year());
      //strncpy(boottime, boottimetemp.c_str(), 20);     // If we got time set boottime
  } else {
      setTime(1609455600);  // 1.1.2021 00:00 Initialize time
  }
  
  // Check state befor sendMQTT()
  measureBatt();                    // check battery voltage
  getRelayState();                  // Check relays status
  millisecs = millis();             // start millisecs
  sendMQTT(MQTT_TRESHOLD);          // Send treshold values
  sendMQTT(MQTT_VERSION);           // Send sketch version
  

  DEBUG_PRINT("Check for Update (otau_enabled): ");
  DEBUG_PRINTLN(otau_enabled);

  if (otau_enabled) { 
    DEBUG_PRINTLN("Check for Update ...");
    checkForUpdates();                                    // Check for Updates on HTTP-Server
  } else {
    DEBUG_PRINTLN("Update disabled ...");
  }

// **************************************************************************
//  Setup: Sleep-Mode
// **************************************************************************
  // Only in Power Supply Mode
  if (powermode) {
    DEBUG_PRINTLN("Startup completed in Power Supply mode ...");
    startMeasure.attach(power_sampling_interval*60, set_measure_flag);  // Start measure periodically each configured interval time (set the flag)
  } else 
    if ((millis() - millisecs) > (mqtt_receive_wait * sensors_count) ){
      sendMQTT(MQTT_TRESHOLD);          // Send treshold values
      DEBUG_PRINTLN("Startup completed in Battery mode and MQTT topic received...");
      DEBUG_PRINTLN("Going into deepsleep mode ...");
      ESP.deepSleep(60000000*bat_sampling_interval);                    // 60s * Value in Minutes from config
      yield();
      delay(200);                                                       // Only to ensure the Deepsleep is working
    } else {
      DEBUG_PRINTLN("Going into loop mode ...");
  }
}

// **************************************************************************
//  Main Loop
// **************************************************************************
void loop(void){
  // Only in Power Supply Mode
//  if (powermode || wakeup) { 
    httpServer.handleClient();
    if ((mqtt_enabled) && (strlen(mqtt_server) > 2)){
      if (!client.connected()) {
        reconnectMQTT();
      }
      client.loop();
      delay(10);                        // Wait for receive messages
    }

    if (func_measure_call) {            // If the call function flag is set by ticker, execute measureFreq()
      outputMoisture();                 // Set dry/wet-Status to Gardena Computer
      getRelayState();                  // Check relays status
      measureBatt();                    // check battery voltage
      sendMQTT(MQTT_TRESHOLD);          // Send treshold values
      for (int i = 0; i < 10; i++) {
         receivedFreqValue[i] = false;   // Reset array for received data only in Power supply mode
      }
      func_measure_call = false;
    }
//  } 
  if (!powermode){ // Battery Mode
    if ((!wakeup) &&  ((millis() - millisecs) > (mqtt_receive_wait * sensors_count) )) {   // If wakeup reset by mqtt and subscribed topic received go sleep
       DEBUG_PRINTLN("Going into deepsleep mode (wakeup: false) ...");
       ESP.deepSleep(60000000*bat_sampling_interval);                    // 60s * Value in Minutes from config
       yield();
       delay(100);                                                       // Only to ensure the Deepsleep is working
    }  
  }
}
// **************************************************************************
//  EOF
// **************************************************************************
