/*
 * Copyright Chris Petrich, Ralph Lautenschläger
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License. 
 *   
 *   File:    discover_fake_DS18B20.ino
 *   Author:  Chris Petrich
 *   Version: 22 Oct 2019 
 *   
 *   Source:  https://github.com/cpetrich/counterfeit_DS18B20/
 *   Documentation:  https://github.com/cpetrich/counterfeit_DS18B20/
 *   
 * This demonstration script performs authenticity tests on DS18B20 sensors by
 * evaluating ROM code and Scratchpad Register. It uses documented ROM commands 
 * F0h and 55h and Function Commands 44h, 4Eh and BEh, only.
 * It does not test the power-up state and it does not write to or test the EEPROM.
 * It is INTENDED for EDUCATIONAL PURPOSES, only.
 * There may be circumstances under which the sketch permanently damages one-wire 
 * sensors in obvious or non-obvious ways.
 * (I don't think it does that to authentic Maxim sensors, but I won't guarantee
 * anything. See licence text for details.)
 * 
 * ----------------------------------------------------------------------------------
 * The Sketch has been adapted for the DS18B20 tester to include 
 * the functions for displaying the results on a 160x128 colour display.
 * 
 *   File:    DS18B20-Test.ino
 *   Author:  Ralph Lautenschlaeger
 *   Version: V 1.0.134 - 02.12.2022
 *  
 */

// Tested with OneWire Version 2.3
// https://github.com/PaulStoffregen/OneWire
#include "OneWire.h"
#include "version.h"

#define pin_onewire 2
#define pinGLED 9
#define pinRLED 5
#define pinBLED 6

#define TFT_PIN_CS   7  // Arduino-Pin an Display CS   
#define TFT_PIN_DC   10 // Arduino-Pin an 
#define TFT_PIN_RST  3  // Arduino Reset-Pin

#define Comm Serial

#include <SPI.h>             // SPI für die Kommunikation
#include <Adafruit_GFX.h>    // Adafruit Grafik-Bibliothek wird benötigt
#include <Adafruit_ST7735.h> // Adafruit ST7735-Bibliothek wird benötigt

const String VERSION = SKETCH_VERSION;

Adafruit_ST7735 tft = Adafruit_ST7735(TFT_PIN_CS, TFT_PIN_DC, TFT_PIN_RST);  // ST7735-Bibliothek Setup


OneWire *ds;

void print_hex(uint8_t value, bool tftwrite = false) {
  if (value < 16) { 
    Comm.write('0');
    if (tftwrite) tft.print('0');
  }
  Comm.print(value, HEX);
  if (tftwrite) tft.print(value, HEX);

}

void print_array(uint16_t cLeft, uint16_t cTop, uint8_t *data, int n, char sep = ',', bool tftwrite = false) {  
  int idx;
  if (tftwrite) tft.setCursor(cLeft,cTop);
  for (idx=0; idx<n; idx++) {
    print_hex(data[idx], tftwrite);
    if (idx != n-1){
      Comm.write(sep);
      if (tftwrite) tft.print(sep);
    }
  }
}

bool read_scratchpad(uint8_t *addr, uint8_t *buff9) {
  ds->reset();
  ds->select(addr);
  ds->write(0xBE); // read scratchpad
  int idx;
  for (idx=0; idx<9; idx++)
    buff9[idx] = ds->read();
  return 0 == OneWire::crc8(buff9, 9);
}

void tftprint(uint16_t cLeft, uint16_t cTop, uint16_t color, String text){
  if ((cLeft >= 0) && (cTop >=0)) {
    // setCursor(links,oben);
    tft.setCursor(cLeft,cTop);
  }
    // setTextColor(farbe);
  tft.setTextColor(color);
  
  // print(text);
  tft.print(text);
}

void setup() {
  String cVersion = "(V." + VERSION + ")";
    
  /***
   * ST7735-Chip initialisieren (INITR_BLACKTAB / INITR_REDTAB / INITR_GREENTAB)
   * Muss bei AZ-Delivery 1.77'' 160x128px RGB TFT INITR_GREENTAB sein ansonsten Pixelfehler rechts und unten.
   * Hinweis: https://github.com/adafruit/Adafruit-ST7735-Library/blob/master/examples/soft_spitftbitmap/soft_spitftbitmap.ino#L52
   * Zeile 52-65  
   ***/
   
  tft.initR(INITR_BLACKTAB);   
      /***  
   * Die tft Funktionen kommen aus der Adafruit Grafik-Bibliothek (Adafruit_GFX) die möglichen Funktionen finden sich unter:
   * https://learn.adafruit.com/adafruit-gfx-graphics-library?view=all
   * bzw.
   * https://cdn-learn.adafruit.com/downloads/pdf/adafruit-gfx-graphics-library.pdf
   ***/
   
  //rotate to landscape
  tft.setRotation(3);
  
  // fillScreen(farbe);
  tft.fillScreen(ST7735_BLACK);

  // drawRect(pos_links,pos_oben,breite,hoehe,farbe);
  tft.drawRect(0,0,160,128,ST7735_RED); 
  
  // setTextSize(groesse);
  tft.setTextSize(1);

  
  Comm.begin(115200);
  Comm.println();  
  
  pinMode(pinRLED, OUTPUT); 
  pinMode(pinGLED, OUTPUT); 
  pinMode(pinBLED, OUTPUT); 
  digitalWrite(pinBLED, HIGH);
 
  ds = new OneWire(pin_onewire);
  
  {
    // output file name without leading path
    char *file = __FILE__;
    int i;
    for (i = strlen(file); i > 0; i--)
      if ((file[i] == '\\') || (file[i] == '/')) {
        i++;
        break;  
      }    
    Comm.print(F("\n--- # "));
    Comm.println(&file[i]);
    tftprint(35, 56, ST7735_WHITE, F("DS18B20-Tester"));
    tftprint(42, 66, ST7735_WHITE, cVersion);
    
  }
  Comm.println(F("This sketch will test DS18B20 sensors attached to"));
  Comm.print(F("  pin "));
  Comm.print(pin_onewire, DEC);
  Comm.println(F(" for differences with Maxim Integrated-produced DS18B20"));
  Comm.println(F("  using only functionality documented in the datasheet and in"));
  Comm.println(F("  Maxim Application Note AN4377."));  
  Comm.println();
  delay(4000);
  digitalWrite(pinBLED, LOW);
}

void loop() {
  
  // ROM address of current sensor
  uint8_t addr[8];  
  // buffers for scratchpad register
  uint8_t buffer0[9];
  uint8_t buffer1[9];
  uint8_t buffer2[9];
  uint8_t buffer3[9];
  // flag to indicate if validation
  //  should be repeated at a different
  //  sensor temperature
 
  ds->reset_search();
  while (ds->search(addr)) {
    bool t_ok;
    int zeile = 5;
    int fake_flags = 0;
    digitalWrite(pinGLED, LOW);
    digitalWrite(pinRLED, LOW);
    digitalWrite(pinBLED, LOW);
    // fillScreen(farbe);
    tft.fillScreen(ST7735_BLACK);

    // drawRect(pos_links,pos_oben,breite,hoehe,farbe);
    tft.drawRect(0,0,160,128,ST7735_RED); 
 
    tft.setTextSize(1);
    tft.setTextColor(ST7735_WHITE);    
    print_array(1, zeile, addr, 8, '-', true);
    zeile += 10;
    if (0 != OneWire::crc8(addr, 8)) {
      // some fake sensors can have their ROM overwritten with
      // arbitrary nonsense, so we don't expect anything good
      // if the ROM doesn't check out
      fake_flags += 1;
      Comm.print(F(" (CRC Error -> Error.)"));
      tftprint(1, zeile, ST7735_WHITE, F("(CRC Error)"));
    }

    if ((addr[6] != 0) || (addr[5] != 0) || (addr[0] != 0x28)) {
      fake_flags += 1;
      Comm.print(F(": ROM does not follow expected pattern 28-xx-xx-xx-xx-00-00-crc. Error."));
      tftprint(67, zeile, ST7735_RED, F(" ROM not ok."));
    } else {
      Comm.print(F(": ROM ok."));
      tftprint(67, zeile, ST7735_GREEN, F(" ROM ok."));
    }    
    Comm.println();
    
    if (!read_scratchpad(addr, buffer0)) read_scratchpad(addr, buffer0);
    zeile += 10;
    Comm.print(F("Scratchpad Register: "));
    tftprint(1, zeile, ST7735_WHITE, F("Scratchpad Register: "));
    zeile += 10;
    print_array(1, zeile , buffer0, 9, '/', true);
    if (0 != OneWire::crc8(buffer0, 9)) {
      // Unlikely that a sensor will mess up the CRC of the scratchpad.
      // --> Assume we're dealing with a bad connection rather than a bad 
      //     sensor, dump data, and move on to next sensor.
      Comm.println(F("  CRC Error. Check connections or replace sensor."));
      tft.fillScreen(ST7735_RED);
      tftprint(50, 50, ST7735_BLACK, F("CRC Error."));
      tftprint(32, 60, ST7735_BLACK, F("Check connections"));
      tftprint(32, 70, ST7735_BLACK, F("or replace sensor."));
      delay(2000);
      continue;      
    }
    Comm.println();
    zeile += 10;

    // Check content of user EEPROM. Since the EEPROM may have been programmed by the user earlier
    // we do not use this as a test. Rather, we dump this as info.
    Comm.print(F("  Info only: Scratchpad bytes 2,3,4 ("));
    print_array(1, zeile, buffer0+2,3,'/', true);
    Comm.print(F("): "));
    zeile += 10;
    if ((buffer0[2] != 0x4b) || (buffer0[3] != 0x46) || (buffer0[4] != 0x7f)){
      Comm.println(F(" not Maxim default values 4B/46/7F."));
      tftprint(1, zeile, ST7735_RED, F("Not Maxim default values."));
    } else {
      Comm.println(F(" Maxim default values."));
      tftprint(1, zeile, ST7735_GREEN, F("Maxim default values."));
    }

    zeile += 10;
    Comm.print(F("  Scratchpad byte 5 (0x"));
    tftprint(1, zeile, ST7735_WHITE, F("Scratchpad #5(0x"));
    print_hex(buffer0[5], true);
    Comm.print(F("): "));
    tftprint(108, zeile, ST7735_WHITE, F("): "));
    if (buffer0[5] != 0xff) {
      fake_flags += 1;
      Comm.println(F(" should have been 0xFF according to datasheet. Error."));
      tftprint(126, zeile, ST7735_RED, F("nOK"));
    } else {
      Comm.println(F(" ok."));
      tftprint(126, zeile, ST7735_GREEN, F("OK"));
    }

    zeile += 10;
    Comm.print(F("  Scratchpad byte 6 (0x"));
    tftprint(1, zeile, ST7735_WHITE, F("Scratchpad #6(0x"));
    print_hex(buffer0[6], true);
    Comm.print(F("): "));
    tftprint(108, zeile, ST7735_WHITE, F("): "));
    if ( ((buffer0[6] == 0x00) || (buffer0[6] > 0x10)) || // totall wrong value
         ( ((buffer0[0] != 0x50) || (buffer0[1] != 0x05)) && ((buffer0[0] != 0xff) || (buffer0[1] != 0x07)) && // check for valid conversion...
           (((buffer0[0] + buffer0[6]) & 0x0f) != 0x00) ) ) { //...before assessing DS18S20 compatibility.
      fake_flags += 1;
      Comm.println(" unexpected value. Error.");
      tftprint(126, zeile, ST7735_RED, F("nOK"));
    } else {
      Comm.println(" ok.");
      tftprint(126, zeile, ST7735_GREEN, F("OK"));
    }  

    zeile += 10;    
    Comm.print(F("  Scratchpad byte 7 (0x"));
    tftprint(1, zeile, ST7735_WHITE, F("Scratchpad #7(0x"));
    print_hex(buffer0[7], true);
    Comm.print(F("): "));
    tftprint(108, zeile, ST7735_WHITE, F("): "));
    if (buffer0[7] != 0x10) {
      fake_flags += 1;
      Comm.println(F(" should have been 0x10 according to datasheet. Error."));
      tftprint(126, zeile, ST7735_RED, F("nOK"));
    } else {
      Comm.println(F(" ok."));
      tftprint(126, zeile, ST7735_GREEN, F("OK"));
    }


/******************************************
 * Next Page
 * ****************************************/
    zeile = 5;

    delay(5000);
    // fillScreen(farbe);
    tft.fillScreen(ST7735_BLACK);

    // drawRect(pos_links,pos_oben,breite,hoehe,farbe);
    tft.drawRect(0,0,160,128,ST7735_RED); 

    // set the resolution to 10 bit and modify alarm registers    
    ds->reset();
    ds->select(addr);
    ds->write(0x4E); // write scratchpad. MUST be followed by 3 bytes as per datasheet.
    ds->write(buffer0[2] ^ 0xff);
    ds->write(buffer0[3] ^ 0xff);
    ds->write(0x3F);
    ds->reset();

    if (!read_scratchpad(addr, buffer1)) read_scratchpad(addr, buffer1);
    
    Comm.print(F("  0x4E modifies alarm registers: "));
    tftprint(1, zeile, ST7735_WHITE, F("mod alarm registers: "));
    if ((buffer1[2] != (buffer0[2] ^ 0xff)) || (buffer1[3] != (buffer0[3] ^ 0xff))) {
      fake_flags += 1;
      Comm.print(F(" cannot modify content as expected (want: "));
      print_hex(buffer0[2] ^ 0xff, false);
      Comm.write('/');
      print_hex(buffer0[3] ^ 0xff, false);
      Comm.print(F(", got: "));
      print_array(-1, -1, buffer1+2, 2, '/', false);
      Comm.println(F("). Error."));      
      tftprint(141, zeile, ST7735_RED, F("nOK"));
    } else {
      Comm.println(F(" ok."));
      tftprint(147, zeile, ST7735_GREEN, F("OK"));
    }

    zeile += 10;    
    Comm.print(F("  0x4E accepts 10 bit resolution: "));
    tftprint(1, zeile, ST7735_WHITE, F("accepts 10 bit res.: "));
    if (buffer1[4] != 0x3f) {
      fake_flags += 1;
      Comm.print(F(" rejected (want: 0x3F, got: "));
      print_hex(buffer1[4], false);
      Comm.println(F("). Error."));
      tftprint(141, zeile, ST7735_RED, F("nOK"));
    } else {
      Comm.println(F(" ok."));
      tftprint(147, zeile, ST7735_GREEN, F("OK"));
    }        

    zeile += 10;    
    Comm.print(F("  0x4E preserves reserved bytes: "));
    tftprint(1, zeile, ST7735_WHITE, F("preserves resvd bytes: "));
    if ((buffer1[5] != buffer0[5]) || (buffer1[6] != buffer0[6]) || (buffer1[7] != buffer0[7])) {
      fake_flags += 1;
      Comm.print(F(" no, got: "));
      print_array(1, 50, buffer1+5, 3, '/', false);
      Comm.println(F(". Error."));
      tftprint(141, zeile, ST7735_RED, F("nOK"));
    } else {
      Comm.println(F(" ok."));    
      tftprint(147, zeile, ST7735_GREEN, F("OK"));
    }      

    // set the resolution to 12 bit
    ds->reset();
    ds->select(addr);
    ds->write(0x4E); // write scratchpad. MUST be followed by 3 bytes as per datasheet.
    ds->write(buffer0[2]);
    ds->write(buffer0[3]);
    ds->write(0x7f);
    ds->reset();

    if (!read_scratchpad(addr, buffer2)) read_scratchpad(addr, buffer2);

    zeile += 10;    
    Comm.print(F("  0x4E accepts 12 bit resolution: "));
    tftprint(1, zeile, ST7735_WHITE, F("accepts 12 bit res.: "));
    if (buffer2[4] != 0x7f) {
      fake_flags += 1;
      Comm.print(F(" rejected (expected: 0x7F, got: "));
      print_hex(buffer2[4], false);
      Comm.println(F("). Error."));
      tftprint(141, zeile, ST7735_RED, F("nOK"));
    } else {
      Comm.println(F(" ok."));
      tftprint(147, zeile, ST7735_GREEN, F("OK"));
    }

    zeile += 10;    
    Comm.print(F("  0x4E preserves reserved bytes: "));
    tftprint(1, zeile, ST7735_WHITE, F("preserves resvd bytes: "));
    if ((buffer2[5] != buffer1[5]) || (buffer2[6] != buffer1[6]) || (buffer2[7] != buffer1[7])) {
      fake_flags += 1;
      Comm.print(F(" no, got: "));
      print_array(1, 60, buffer2+5, 3, '/', false);
      Comm.println(F(". Error."));
      tftprint(141, zeile, ST7735_RED, F("nOK"));
    } else {
      Comm.println(F(" ok."));
      tftprint(147, zeile, ST7735_GREEN, F("OK"));
    }

    zeile += 10;    
    Comm.print("  Checking byte 6 upon temperature change: ");
    tftprint(1, zeile, ST7735_WHITE, F("Check. #6 temp. chge: "));
    if (( ((buffer2[0] == 0x50) && (buffer2[1] == 0x05)) || ((buffer2[0] == 0xff) && (buffer2[1] == 0x07)) ||
         ((buffer2[6] == 0x0c) && (((buffer2[0] + buffer2[6]) & 0x0f) == 0x00)) ) &&
         ((buffer2[6] >= 0x00) && (buffer2[6] <= 0x10)) ){
      // byte 6 checked out as correct in the initial test but the test ambiguous.
      //   we need to check if byte 6 is consistent after temperature conversion
            
      // We'll do a few temperature conversions in a row.
      // Usually, the temperature rises slightly if we do back-to-back
      //   conversions.
      int count = 5;
      do {
        count -- ;
        if (count < 0)
          break;
        // perform temperature conversion
        ds->reset();
        ds->select(addr);
        ds->write(0x44);
        delay(750);
        
        if (!read_scratchpad(addr, buffer3)) read_scratchpad(addr, buffer3);
        
      } while ( ((buffer3[0] == 0x50) && (buffer3[1] == 0x05)) || ((buffer3[0] == 0xff) && (buffer3[1] == 0x07)) ||
                ((buffer3[6] == 0x0c) && (((buffer3[0] + buffer3[6]) & 0x0f) == 0x00)) );
      if (count < 0) {
        Comm.println(F(" Inconclusive. Please change sensor temperature and repeat."));
        tftprint(132, zeile, ST7735_YELLOW, F("Inc."));
        t_ok = false;
      } else {
        t_ok = true;
        if ((buffer3[6] != 0x0c) && (((buffer3[0] + buffer3[6]) & 0x0f) == 0x00)) {
          Comm.println(F(" ok."));
          tftprint(141, zeile, ST7735_GREEN, F(" Ok"));
        } else {
          fake_flags += 1;
          Comm.print(F(" Temperature LSB = 0x"));
          print_hex(buffer3[0], false);
          Comm.print(F(" but byte 6 = 0x"));
          print_hex(buffer3[6], false);
          Comm.println(F(". Error."));
          tftprint(147, zeile, ST7735_RED, F(" nOk"));
        }
      }
    } else {
      Comm.println(F("not necessary. Skipped."));
      tftprint(132, zeile, ST7735_YELLOW, F("Skip"));
      t_ok = true;
    }

    //zeile += 20;    
    Comm.print(F("  --> "));
    if (!t_ok) {
      Comm.print(F("DS18S20 counterfeit test not completed, otherwise sensor"));
    } else 
      Comm.print(F("Sensor"));

    tft.setTextSize(2);
    
    if (fake_flags == 0) {
      Comm.println(F(" responded like a genuie Maxim."));
      Comm.println(F("      Not tested: EEPROM, Parasite Power, and undocumented commands."));
      tft.drawRect(30,80,100,20,ST7735_WHITE); 
      tft.fillRect(31,81,98,18,ST7735_GREEN);
      tftprint(68, 83, ST7735_BLACK, F("OK"));
    } else {
      Comm.print(F(" appears to be counterfeit based on "));
      Comm.print(fake_flags, DEC);
      Comm.println(F(" deviations."));
      tft.drawRect(30,80,100,20,ST7735_WHITE); 
      tft.fillRect(31,81,98,18,ST7735_RED);
      tftprint(44, 83, ST7735_BLACK, F("not OK"));
      if (fake_flags == 1) {
        Comm.println(F("  The number of deviations is unexpectedly small."));
        Comm.println(F("  Please see https://github.com/cpetrich/counterfeit_DS18B20/"));
        Comm.println(F("  to help interpret the result."));
        tft.drawRect(30,80,100,20,ST7735_WHITE); 
        tft.fillRect(31,81,98,18,ST7735_YELLOW);
        tftprint(32, 83, ST7735_BLACK, F("maybe OK"));
      }
    }
    Comm.println();

    if (fake_flags == 0) {
      digitalWrite(pinGLED, HIGH);
    } else if (fake_flags == 1){
      digitalWrite(pinGLED, HIGH);
      digitalWrite(pinRLED, HIGH);
    } else {
      digitalWrite(pinRLED, HIGH);
    }  
    delay(5000);
  } // done with all sensors
  
  // No DS18B20 is connected or device demaged
  if (!ds->search(addr)) {
      digitalWrite(pinGLED, HIGH);
      digitalWrite(pinRLED, HIGH);
      digitalWrite(pinBLED, HIGH);
      tft.drawRect(15,58,130,20,ST7735_WHITE); 
      tft.fillRect(16,59,128,18,ST7735_RED);
      tftprint(20, 60, ST7735_WHITE, F("No DS18B20 connected"));
      tftprint(20, 69, ST7735_WHITE, F("or no prasite power"));
      delay(1000);
      digitalWrite(pinGLED, LOW);
      digitalWrite(pinRLED, LOW);
      digitalWrite(pinBLED, LOW);
      delay(1000);
  }
  Comm.println(F("------------------------------------------------"));
}
